/*
 * Decompiled with CFR 0.152.
 */
package io.ballerina.messaging.broker.core.transaction;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.ballerina.messaging.broker.common.ValidationException;
import io.ballerina.messaging.broker.core.BrokerException;
import io.ballerina.messaging.broker.core.store.MessageStore;
import io.ballerina.messaging.broker.core.transaction.Branch;
import io.ballerina.messaging.broker.core.transaction.BranchFactory;
import io.ballerina.messaging.broker.core.transaction.DtxStateTransitionException;
import io.ballerina.messaging.broker.core.transaction.UnknownDtxBranchException;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.ThreadSafe;
import javax.transaction.xa.Xid;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class Registry {
    private static final Logger LOGGER = LoggerFactory.getLogger(Registry.class);
    private static final String ASSOCIATED_XID_ERROR_MSG = "Branch still has associated active sessions for xid ";
    private static final String TIMED_OUT_ERROR_MSG = "Transaction timed out for xid ";
    private final Map<Xid, Branch> branchMap;
    private final ScheduledExecutorService branchTimeoutExecutorService;
    private final BranchFactory branchFactory;
    private final Set<Xid> storedXidSet;

    Registry(BranchFactory branchFactory) {
        this.branchFactory = branchFactory;
        this.branchMap = new ConcurrentHashMap<Xid, Branch>();
        this.storedXidSet = ConcurrentHashMap.newKeySet();
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("DtxBranchTimeoutExecutor-%d").build();
        this.branchTimeoutExecutorService = Executors.newSingleThreadScheduledExecutor(threadFactory);
    }

    public void register(Branch branch) throws ValidationException {
        if (Objects.nonNull(this.branchMap.putIfAbsent(branch.getXid(), branch))) {
            throw new ValidationException("Branch with the same xid " + branch.getXid() + " is already registered.");
        }
    }

    public void unregister(Xid xid) {
        if (Objects.isNull(this.branchMap.remove(xid))) {
            this.storedXidSet.remove(xid);
        }
    }

    public Branch getBranch(Xid xid) throws ValidationException {
        if (this.storedXidSet.contains(xid)) {
            throw new ValidationException("Branch is in prepared stage. Branch can be only be committed or rollbacked.");
        }
        return this.branchMap.get(xid);
    }

    public synchronized void prepare(Xid xid) throws ValidationException, BrokerException {
        if (this.storedXidSet.contains(xid)) {
            throw new DtxStateTransitionException(xid, Branch.State.PREPARED, Branch.State.PREPARED);
        }
        Branch branch = this.branchMap.get(xid);
        if (Objects.isNull(branch)) {
            throw new ValidationException("Branch not found with xid " + xid);
        }
        if (branch.hasAssociatedActiveSessions()) {
            throw new ValidationException(ASSOCIATED_XID_ERROR_MSG + xid);
        }
        this.checkForBranchExpiration(branch);
        branch.clearAssociations();
        if (branch.getState() == Branch.State.ROLLBACK_ONLY) {
            throw new ValidationException("Transaction can only be rollbacked");
        }
        if (branch.getState() != Branch.State.ACTIVE) {
            throw new ValidationException("Cannot prepare a branch in state " + (Object)((Object)branch.getState()));
        }
        branch.prepare();
    }

    private void checkForBranchExpiration(Branch branch) throws ValidationException {
        if (branch.isExpired() || !this.cancelTimeoutTask(branch)) {
            this.unregister(branch.getXid());
            throw new ValidationException(TIMED_OUT_ERROR_MSG + branch.getXid());
        }
    }

    public synchronized void commit(Xid xid, boolean onePhase) throws ValidationException, BrokerException {
        Branch branch = this.branchMap.get(xid);
        if (Objects.isNull(branch)) {
            branch = this.checkForBranchRecovery(xid);
        } else {
            if (branch.hasAssociatedActiveSessions()) {
                throw new ValidationException(ASSOCIATED_XID_ERROR_MSG + xid);
            }
            this.checkForBranchExpiration(branch);
            if (branch.isRollbackOnly()) {
                throw new ValidationException("Branch is set to rollback only. Can't commit with xid " + xid);
            }
            if (!onePhase && !branch.isPrepared()) {
                throw new ValidationException("Cannot call two-phase commit on a non-prepared branch for xid " + xid);
            }
        }
        if (onePhase && branch.isPrepared()) {
            throw new ValidationException("Cannot call one-phase commit on a prepared branch for xid " + xid);
        }
        branch.clearAssociations();
        branch.commit(onePhase);
        branch.setState(Branch.State.FORGOTTEN);
        this.unregister(xid);
    }

    private Branch checkForBranchRecovery(Xid xid) throws UnknownDtxBranchException {
        if (!this.storedXidSet.contains(xid)) {
            throw new UnknownDtxBranchException(xid);
        }
        Branch branch = this.branchFactory.createBranch(xid);
        branch.markAsRecoveryBranch();
        return branch;
    }

    private boolean cancelTimeoutTask(Branch branch) {
        Future timeoutTaskFuture = branch.getTimeoutTaskFuture();
        return Objects.isNull(timeoutTaskFuture) || timeoutTaskFuture.isCancelled() || timeoutTaskFuture.cancel(false);
    }

    public synchronized void rollback(Xid xid) throws ValidationException, BrokerException {
        Branch branch = this.branchMap.get(xid);
        if (Objects.isNull(branch)) {
            branch = this.checkForBranchRecovery(xid);
        } else {
            this.checkForBranchExpiration(branch);
            if (branch.hasAssociatedActiveSessions()) {
                throw new ValidationException(ASSOCIATED_XID_ERROR_MSG + xid);
            }
            branch.clearAssociations();
        }
        branch.dtxRollback();
        branch.setState(Branch.State.FORGOTTEN);
        this.unregister(xid);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void forget(Xid xid) throws ValidationException {
        Branch branch = this.branchMap.get(xid);
        if (Objects.isNull(branch)) {
            throw new ValidationException("Branch not found with xid " + xid);
        }
        Branch branch2 = branch;
        synchronized (branch2) {
            if (branch.hasAssociatedActiveSessions()) {
                throw new ValidationException(ASSOCIATED_XID_ERROR_MSG + xid);
            }
            if (branch.getState() != Branch.State.HEUR_COM && branch.getState() != Branch.State.HEUR_RB) {
                throw new ValidationException("Branch is not heuristically complete, hence unable to forget. Xid " + xid);
            }
            branch.setState(Branch.State.FORGOTTEN);
            this.unregister(xid);
        }
    }

    public void setTimeout(Xid xid, long timeout, TimeUnit timeUnit) throws ValidationException {
        Branch branch = this.branchMap.get(xid);
        if (Objects.isNull(branch)) {
            throw new ValidationException("Branch not found with xid " + xid);
        }
        if (timeout == 0L) {
            return;
        }
        ScheduledFuture<?> future = this.branchTimeoutExecutorService.schedule(() -> {
            LOGGER.debug("timing out dtx task with xid {}", (Object)xid);
            Branch branch2 = branch;
            synchronized (branch2) {
                if (branch.isPrepared()) {
                    LOGGER.debug("Branch already prepared. Won't be timed out. Xid {}", (Object)xid);
                    return;
                }
                try {
                    this.rollback(xid);
                    branch.setState(Branch.State.TIMED_OUT);
                }
                catch (ValidationException | BrokerException e) {
                    LOGGER.error("Error occurred while rolling back timed out branch with Xid " + xid, e);
                }
            }
        }, timeout, timeUnit);
        branch.setTimeoutTaskFuture(future);
    }

    void syncWithMessageStore(MessageStore messageStore) throws BrokerException {
        this.storedXidSet.clear();
        messageStore.retrieveStoredXids(this.storedXidSet::add);
    }
}

