/*
 * Decompiled with CFR 0.152.
 */
package io.ballerina.messaging.broker.core;

import com.google.common.collect.Iterables;
import io.ballerina.messaging.broker.common.ValidationException;
import io.ballerina.messaging.broker.common.util.function.ThrowingConsumer;
import io.ballerina.messaging.broker.core.Binding;
import io.ballerina.messaging.broker.core.BrokerException;
import io.ballerina.messaging.broker.core.Consumer;
import io.ballerina.messaging.broker.core.CyclicConsumerIterator;
import io.ballerina.messaging.broker.core.DetachableMessage;
import io.ballerina.messaging.broker.core.Message;
import io.ballerina.messaging.broker.core.Queue;
import io.ballerina.messaging.broker.core.metrics.BrokerMetricManager;
import io.ballerina.messaging.broker.core.queue.MemQueueImpl;
import io.ballerina.messaging.broker.core.queue.UnmodifiableQueueWrapper;
import io.ballerina.messaging.broker.core.util.MessageTracer;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import javax.transaction.xa.Xid;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class QueueHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(QueueHandler.class);
    private Queue queue;
    private final CyclicConsumerIterator consumerIterator;
    private final Queue redeliveryQueue;
    private final BrokerMetricManager metricManager;
    private final Set<Consumer> consumers;
    private final Queue unmodifiableQueueView;
    private final Map<Binding, ThrowingConsumer<Binding, BrokerException>> bindingChangeListenersMap;

    QueueHandler(Queue queue, BrokerMetricManager metricManager) {
        this.queue = queue;
        queue.setQueueHandler(this);
        this.unmodifiableQueueView = new UnmodifiableQueueWrapper(queue);
        this.redeliveryQueue = new MemQueueImpl(queue.getName(), false);
        this.metricManager = metricManager;
        this.consumers = ConcurrentHashMap.newKeySet();
        this.consumerIterator = new CyclicConsumerIterator();
        this.bindingChangeListenersMap = new ConcurrentHashMap<Binding, ThrowingConsumer<Binding, BrokerException>>();
    }

    public Queue getUnmodifiableQueue() {
        return this.unmodifiableQueueView;
    }

    public Collection<Consumer> getConsumers() {
        return Collections.unmodifiableCollection(this.consumers);
    }

    boolean addConsumer(Consumer consumer) {
        return this.consumers.add(consumer);
    }

    boolean removeConsumer(Consumer consumer) {
        return this.consumers.remove(consumer);
    }

    void enqueue(Message message) throws BrokerException {
        boolean success;
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Enqueuing message {} to queue {}", (Object)message, (Object)this.queue.getName());
        }
        if (success = this.queue.enqueue(message)) {
            this.metricManager.addInMemoryMessage();
            MessageTracer.trace(message, this, "Successfully enqueued message to queue.");
        } else {
            message.release();
            MessageTracer.trace(message, this, "Failed to enqueue. Message dropped!");
            LOGGER.info("Failed to publish message {} to the queue {}", (Object)message, (Object)this.queue.getName());
        }
    }

    void prepareForEnqueue(Xid xid, Message message) throws BrokerException {
        MessageTracer.trace(message, xid, this, "Prepare to enqueue message.");
        this.queue.prepareEnqueue(xid, message);
    }

    void prepareForDetach(Xid xid, DetachableMessage detachableMessage) throws BrokerException {
        MessageTracer.trace(detachableMessage, xid, this, "Prepare dequeue message event.");
        this.queue.prepareDetach(xid, detachableMessage);
    }

    public void commit(Xid xid) {
        this.queue.commit(xid);
        MessageTracer.trace(xid, this, "Committed prepared events on queue.");
    }

    public void rollback(Xid xid) {
        this.queue.rollback(xid);
        MessageTracer.trace(xid, this, "Rollbacked prepared events on queue.");
    }

    Message takeForDelivery() {
        Message message = this.redeliveryQueue.dequeue();
        if (message == null) {
            message = this.queue.dequeue();
            MessageTracer.trace(message, this, "Dequeue message for delivery.");
        } else {
            MessageTracer.trace(message, this, "Dequeue message for re-delivery.");
        }
        return message;
    }

    void dequeue(DetachableMessage detachableMessage) throws BrokerException {
        this.queue.detach(detachableMessage);
        this.metricManager.removeInMemoryMessage();
        MessageTracer.trace(detachableMessage, this, "Acknowledge message.");
    }

    public void requeue(Message message) throws BrokerException {
        boolean success = this.redeliveryQueue.enqueue(message);
        if (!success) {
            LOGGER.warn("Enqueuing message since redelivery queue for {} is full. message:{}", (Object)this.queue.getName(), (Object)message);
            this.enqueue(message);
        }
        MessageTracer.trace(message, this, "Requeue message.");
    }

    CyclicConsumerIterator getCyclicConsumerIterator() {
        this.consumerIterator.setIterator(Iterables.cycle(this.consumers).iterator());
        return this.consumerIterator;
    }

    boolean isEmpty() {
        return this.size() == 0;
    }

    public int size() {
        return this.queue.size() + this.redeliveryQueue.size();
    }

    boolean isUnused() {
        return this.consumers.isEmpty();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeAllConsumers() {
        Iterator<Consumer> iterator = this.consumers.iterator();
        while (iterator.hasNext()) {
            Consumer consumer = iterator.next();
            try {
                consumer.close();
            }
            catch (BrokerException e) {
                LOGGER.error("Error occurred while closing the consumer [ " + consumer + " ] for queue [ " + this.queue.toString() + " ]", (Throwable)e);
            }
            finally {
                iterator.remove();
            }
        }
    }

    public int consumerCount() {
        return this.consumers.size();
    }

    public void addBinding(Binding binding, ThrowingConsumer<Binding, BrokerException> bindingChangeListener) {
        this.bindingChangeListenersMap.put(binding, bindingChangeListener);
    }

    public int releaseResources() throws BrokerException {
        this.closeAllConsumers();
        for (Map.Entry<Binding, ThrowingConsumer<Binding, BrokerException>> entry : this.bindingChangeListenersMap.entrySet()) {
            entry.getValue().accept((Object)entry.getKey());
        }
        return this.redeliveryQueue.clear() + this.queue.clear();
    }

    public void removeBinding(Binding binding) {
        this.bindingChangeListenersMap.remove(binding);
    }

    public int purgeQueue() throws ValidationException {
        if (this.consumerCount() == 0) {
            int queueMessages = this.queue.clear();
            int totalMessages = queueMessages + this.redeliveryQueue.size();
            this.redeliveryQueue.clear();
            return totalMessages;
        }
        throw new ValidationException("Cannot purge queue " + this.queue.getName() + " since there " + this.consumerCount() + " active consumer(s)");
    }
}

