/*
 * Decompiled with CFR 0.152.
 */
package org.ballerinax.docker.generator;

import com.github.dockerjava.api.DockerClient;
import com.github.dockerjava.api.model.BuildResponseItem;
import com.github.dockerjava.api.model.PushResponseItem;
import com.github.dockerjava.api.model.ResponseItem;
import com.github.dockerjava.core.DefaultDockerClientConfig;
import com.github.dockerjava.core.DockerClientBuilder;
import com.github.dockerjava.core.LocalDirectorySSLConfig;
import com.github.dockerjava.core.RemoteApiVersion;
import com.github.dockerjava.core.command.BuildImageResultCallback;
import com.github.dockerjava.core.command.PushImageResultCallback;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.Locale;
import org.apache.commons.io.FilenameUtils;
import org.ballerinax.docker.generator.exceptions.DockerGenException;
import org.ballerinax.docker.generator.models.CopyFileModel;
import org.ballerinax.docker.generator.models.DockerModel;
import org.ballerinax.docker.generator.utils.DockerGenUtils;
import org.ballerinax.docker.generator.utils.DockerImageName;

public class DockerArtifactHandler {
    private static final boolean WINDOWS_BUILD = Boolean.parseBoolean(System.getenv("BAL_DOCKER_WINDOWS"));
    private final DockerError dockerBuildError = new DockerError();
    private final DockerError dockerPushError = new DockerError();
    private DockerModel dockerModel;
    private DockerClient dockerClient;
    private DefaultDockerClientConfig dockerClientConfig;

    public DockerArtifactHandler(DockerModel dockerModel) {
        String registry = dockerModel.getRegistry();
        String imageName = dockerModel.getName();
        imageName = !DockerGenUtils.isBlank(registry) ? registry + "/" + imageName + ":" + dockerModel.getTag() : imageName + ":" + dockerModel.getTag();
        dockerModel.setName(imageName);
        this.dockerModel = dockerModel;
        this.dockerClient = this.createClient();
    }

    public void createArtifacts(PrintStream outStream, String logAppender, Path uberJarFilePath, Path outputDir) throws DockerGenException {
        String dockerContent = !WINDOWS_BUILD ? this.generateDockerfile() : this.generateDockerfileForWindows();
        try {
            String logStepCount = this.dockerModel.isBuildImage() ? (this.dockerModel.isPush() ? "3" : "2") : "1";
            outStream.print(logAppender + " - complete 0/" + logStepCount + " \r");
            DockerGenUtils.writeToFile(dockerContent, outputDir.resolve("Dockerfile"));
            outStream.print(logAppender + " - complete 1/" + logStepCount + " \r");
            Path uberJarLocation = outputDir.resolve(DockerGenUtils.extractUberJarName(uberJarFilePath) + ".jar");
            DockerGenUtils.copyFileOrDirectory(uberJarFilePath, uberJarLocation);
            for (CopyFileModel copyFileModel : this.dockerModel.getCopyFiles()) {
                Path target = outputDir.resolve(Paths.get(copyFileModel.getSource(), new String[0]).getFileName());
                Path sourcePath = Paths.get(copyFileModel.getSource(), new String[0]);
                if (!sourcePath.isAbsolute()) {
                    sourcePath = sourcePath.toAbsolutePath();
                }
                DockerGenUtils.copyFileOrDirectory(sourcePath, target);
            }
            if (this.dockerModel.isBuildImage()) {
                this.buildImage(outputDir);
                outStream.print(logAppender + " - complete 2/" + logStepCount + " \r");
                Files.delete(uberJarLocation);
                if (this.dockerModel.isPush()) {
                    this.pushImage();
                    outStream.print(logAppender + " - complete 3/" + logStepCount + " \r");
                }
            }
        }
        catch (IOException e) {
            throw new DockerGenException("unable to write content to " + outputDir);
        }
        catch (InterruptedException e) {
            throw new DockerGenException("unable to create Docker images " + e.getMessage());
        }
    }

    private DockerClient createClient() {
        DefaultDockerClientConfig.Builder dockerClientConfig = DefaultDockerClientConfig.createDefaultConfigBuilder();
        if (System.getProperty("os.name").toLowerCase(Locale.getDefault()).contains("win")) {
            dockerClientConfig.withDockerHost("tcp://localhost:2375");
        }
        if (null != this.dockerModel.getDockerHost()) {
            dockerClientConfig.withDockerHost(this.dockerModel.getDockerHost());
        }
        if (null != this.dockerModel.getDockerCertPath()) {
            dockerClientConfig.withDockerCertPath(this.dockerModel.getDockerCertPath());
        }
        if (null != this.dockerModel.getDockerAPIVersion()) {
            dockerClientConfig.withApiVersion(this.dockerModel.getDockerAPIVersion());
        }
        if (null != this.dockerModel.getRegistry()) {
            dockerClientConfig.withRegistryUrl(this.dockerModel.getRegistry());
        }
        if (null != this.dockerModel.getUsername()) {
            dockerClientConfig.withRegistryUsername(this.dockerModel.getUsername());
        }
        if (null != this.dockerModel.getPassword()) {
            dockerClientConfig.withRegistryPassword(this.dockerModel.getPassword());
        }
        if (null != this.dockerModel.getDockerConfig()) {
            dockerClientConfig.withDockerConfig(this.dockerModel.getDockerConfig());
        }
        this.dockerClientConfig = dockerClientConfig.build();
        DockerGenUtils.printDebug("docker client host: " + this.dockerClientConfig.getDockerHost());
        if (!this.dockerClientConfig.getApiVersion().equals(RemoteApiVersion.unknown())) {
            DockerGenUtils.printDebug("docker client API version: " + this.dockerClientConfig.getApiVersion().getVersion());
        } else {
            DockerGenUtils.printDebug("docker client API version: not-set");
        }
        if (null != this.dockerClientConfig.getSSLConfig() && this.dockerClientConfig.getSSLConfig() instanceof LocalDirectorySSLConfig) {
            LocalDirectorySSLConfig sslConfig = (LocalDirectorySSLConfig)this.dockerClientConfig.getSSLConfig();
            DockerGenUtils.printDebug("docker client certs path: " + sslConfig.getDockerCertPath());
            DockerGenUtils.printDebug("docker client TLS verify: true");
        } else {
            DockerGenUtils.printDebug("docker client TLS verify: false");
        }
        return DockerClientBuilder.getInstance(dockerClientConfig).build();
    }

    public void buildImage(Path dockerDir) throws DockerGenException {
        DockerImageName.validate(this.dockerModel.getName());
        DockerGenUtils.printDebug("building docker image `" + this.dockerModel.getName() + "` from directory `" + dockerDir + "`.");
        try {
            this.dockerClient.buildImageCmd(dockerDir.toFile()).withNoCache(true).withForcerm(true).withTags(Collections.singleton(this.dockerModel.getName())).exec(new DockerBuildImageCallback()).awaitImageId();
        }
        catch (RuntimeException ex) {
            if (ex.getMessage().contains("java.net.SocketException: Connection refused")) {
                this.dockerBuildError.setErrorMsg("unable to connect to docker host: " + this.dockerClientConfig.getDockerHost());
            }
            this.dockerBuildError.setErrorMsg("unable to build docker image: " + DockerGenUtils.cleanErrorMessage(ex.getMessage()));
        }
        this.handleError(this.dockerBuildError);
    }

    public void pushImage() throws InterruptedException, DockerGenException {
        DockerGenUtils.printDebug("pushing docker image `" + this.dockerModel.getName() + "`.");
        try {
            this.dockerClient.pushImageCmd(this.dockerModel.getName()).exec(new DockerImagePushCallback()).awaitCompletion();
        }
        catch (RuntimeException ex) {
            this.dockerBuildError.setErrorMsg(DockerGenUtils.cleanErrorMessage(ex.getMessage()));
        }
        this.handleError(this.dockerPushError);
    }

    private void handleError(DockerError dockerError) throws DockerGenException {
        if (dockerError.isError()) {
            throw new DockerGenException(dockerError.getErrorMsg());
        }
    }

    private String generateDockerfile() {
        StringBuilder dockerfileContent = new StringBuilder();
        dockerfileContent.append("# Auto Generated Dockerfile\n");
        dockerfileContent.append("FROM ").append(this.dockerModel.getBaseImage()).append("\n");
        dockerfileContent.append("\n");
        dockerfileContent.append("LABEL maintainer=\"dev@ballerina.io\"").append("\n");
        dockerfileContent.append("\n");
        if (this.dockerModel.getBaseImage().equals("ballerina/jre8:v1")) {
            dockerfileContent.append("RUN addgroup troupe \\").append("\n");
            dockerfileContent.append("    && adduser -S -s /bin/bash -g 'ballerina' -G troupe -D ballerina \\").append("\n");
            dockerfileContent.append("    && apk add --update --no-cache bash \\").append("\n");
            dockerfileContent.append("    && chown -R ballerina:troupe /usr/bin/java \\").append("\n");
            dockerfileContent.append("    && rm -rf /var/cache/apk/*").append("\n");
            dockerfileContent.append("\n");
        }
        dockerfileContent.append("WORKDIR /home/ballerina").append("\n");
        dockerfileContent.append("\n");
        dockerfileContent.append("COPY ").append(this.dockerModel.getUberJarFileName()).append(" /home/ballerina").append("\n");
        this.dockerModel.getEnv().forEach((key, value) -> dockerfileContent.append("ENV ").append((String)key).append("=").append((String)value).append("\n"));
        this.dockerModel.getCopyFiles().forEach(file -> {
            String sourceFileName = String.valueOf(Paths.get(file.getSource(), new String[0]).getFileName());
            dockerfileContent.append("COPY ").append(sourceFileName).append(" ").append(file.getTarget()).append("\n");
        });
        dockerfileContent.append("\n");
        if (this.dockerModel.isService() && this.dockerModel.getPorts().size() > 0) {
            dockerfileContent.append("EXPOSE ");
            this.dockerModel.getPorts().forEach(port -> dockerfileContent.append(" ").append(port));
        }
        dockerfileContent.append("\n");
        if (this.dockerModel.getBaseImage().equals("ballerina/jre8:v1")) {
            dockerfileContent.append("USER ballerina").append("\n");
            dockerfileContent.append("\n");
        }
        return this.appendCMD(dockerfileContent);
    }

    private String generateDockerfileForWindows() {
        String dockerBase = "# Auto Generated Dockerfile\n\nFROM " + this.dockerModel.getBaseImage() + "\nLABEL maintainer=\"dev@ballerina.io\"\n\nCOPY " + this.dockerModel.getUberJarFileName() + " C:\\\\ballerina\\\\home \n\n";
        StringBuilder stringBuilder = new StringBuilder(dockerBase);
        this.dockerModel.getCopyFiles().forEach(file -> {
            String sourceFileName = String.valueOf(Paths.get(file.getSource(), new String[0]).getFileName());
            stringBuilder.append("COPY ").append(FilenameUtils.separatorsToWindows(sourceFileName)).append(" ").append(FilenameUtils.separatorsToWindows(file.getTarget())).append("\n");
        });
        this.dockerModel.getEnv().forEach((key, value) -> stringBuilder.append("ENV ").append((String)key).append("=").append((String)value));
        if (this.dockerModel.isService() && this.dockerModel.getPorts().size() > 0) {
            stringBuilder.append("EXPOSE ");
            this.dockerModel.getPorts().forEach(port -> stringBuilder.append(" ").append(port));
        }
        stringBuilder.append("\n");
        return this.appendCMD(stringBuilder);
    }

    private String appendCMD(StringBuilder stringBuilder) {
        if (null == this.dockerModel.getCmd() || "".equals(this.dockerModel.getCmd())) {
            if (this.dockerModel.isEnableDebug()) {
                stringBuilder.append("CMD java -Xdebug -Xnoagent -Djava.compiler=NONE -Xrunjdwp:transport=dt_socket,server=y,suspend=y,address=").append(this.dockerModel.getDebugPort()).append(" -jar ").append(this.dockerModel.getUberJarFileName());
            } else {
                stringBuilder.append("CMD java -jar ").append(this.dockerModel.getUberJarFileName());
            }
        } else {
            stringBuilder.append(this.dockerModel.getCmd());
        }
        if (!DockerGenUtils.isBlank(this.dockerModel.getCommandArg())) {
            stringBuilder.append(this.dockerModel.getCommandArg());
        }
        stringBuilder.append("\n");
        return stringBuilder.toString();
    }

    private class DockerImagePushCallback
    extends PushImageResultCallback {
        private DockerImagePushCallback() {
        }

        @Override
        public void onNext(PushResponseItem item) {
            String idLog;
            String statusLog;
            String streamLog;
            if (item.isErrorIndicated()) {
                StringBuilder errString = new StringBuilder("[push][error]: ");
                ResponseItem.ErrorDetail errDetail = null;
                if (null != item.getErrorDetail()) {
                    errDetail = item.getErrorDetail();
                }
                if (null != errDetail && null != errDetail.getCode()) {
                    errString.append("(").append(errDetail.getCode()).append(") ");
                }
                if (null != errDetail && null != errDetail.getMessage()) {
                    errString.append(errDetail.getMessage());
                }
                String errorMessage = errString.toString();
                DockerGenUtils.printDebug(errorMessage);
                DockerArtifactHandler.this.dockerPushError.setErrorMsg("unable to push docker image: " + errorMessage);
            }
            if (null != (streamLog = item.getStream()) && !"".equals(streamLog.replaceAll("\n", ""))) {
                DockerGenUtils.printDebug("[push][stream] " + streamLog.replaceAll("\n", ""));
            }
            if (null != (statusLog = item.getStatus()) && !"".equals(statusLog.replaceAll("\n", ""))) {
                DockerGenUtils.printDebug("[push][status] " + statusLog.replaceAll("\n", ""));
            }
            if (null != (idLog = item.getId())) {
                DockerGenUtils.printDebug("[push][ID]: " + idLog);
            }
            super.onNext(item);
        }
    }

    private class DockerBuildImageCallback
    extends BuildImageResultCallback {
        private DockerBuildImageCallback() {
        }

        @Override
        public void onNext(BuildResponseItem item) {
            String imageIdLog;
            String statusLog;
            String streamLog;
            if (item.isErrorIndicated()) {
                StringBuilder errString = new StringBuilder("[build][error]: ");
                ResponseItem.ErrorDetail errDetail = item.getErrorDetail();
                if (null != errDetail && null != errDetail.getCode()) {
                    errString.append("(").append(errDetail.getCode()).append(") ");
                }
                if (null != errDetail && null != errDetail.getMessage()) {
                    errString.append(errDetail.getMessage());
                }
                String errorMessage = errString.toString();
                DockerGenUtils.printDebug(errorMessage);
                DockerArtifactHandler.this.dockerBuildError.setErrorMsg("unable to build docker image: " + errorMessage);
            }
            if (null != (streamLog = item.getStream()) && !"".equals(streamLog.replaceAll("\n", ""))) {
                DockerGenUtils.printDebug("[build][stream] " + streamLog.replaceAll("\n", ""));
            }
            if (null != (statusLog = item.getStatus()) && !"".equals(statusLog.replaceAll("\n", ""))) {
                DockerGenUtils.printDebug("[build][status] " + statusLog.replaceAll("\n", ""));
            }
            if (null != (imageIdLog = item.getImageId())) {
                DockerGenUtils.printDebug("[build][image-id]: " + imageIdLog);
            }
            super.onNext(item);
        }
    }

    private static class DockerError {
        private boolean error = false;
        private String errorMsg;

        DockerError() {
        }

        boolean isError() {
            return this.error;
        }

        String getErrorMsg() {
            return this.errorMsg;
        }

        void setErrorMsg(String errorMsg) {
            this.error = true;
            this.errorMsg = errorMsg;
        }
    }
}

