// Copyright (c) 2019 WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
//
// WSO2 Inc. licenses this file to you under the Apache License,
// Version 2.0 (the "License"); you may not use this file except
// in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

import ballerina/java;

# The Ballerina interface to provide AMQP Channel related functionality.
public type Channel client object {

    handle amqpChannel;

    # Initializes a `rabbitmq:Channel` object with the given `rabbitmq:Connection` object or connection configuration.
    # Creates a `rabbitmq:Connection` object if only the connection configuration is given.
    #
    # + connectionOrConnectionConfig - A `rabbitmq:Connection` object or a connection configuration
    public function __init(ConnectionConfiguration|Connection connectionOrConnectionConfig) {
        Connection connection = (connectionOrConnectionConfig is Connection) ?
                                connectionOrConnectionConfig : new Connection(connectionOrConnectionConfig);
        self.amqpChannel = createChannel(connection.amqpConnection);
    }

# Declares a non-exclusive, auto-delete, or non-durable queue with the given configurations.
# ```ballerina
# string|rabbitmq:Error? queueResult = newChannel->queueDeclare();
# ```
#
# + queueConfig - Configurations required to declare a queue
# + return - The name of the queue if autogenerated, `()` if the queue was successfully
#            generated with the given parameters, or else a `rabbitmq:Error` if an I/O error is encountered
    public remote function queueDeclare(QueueConfiguration? queueConfig = ()) returns string | Error? {
        var result = nativeQueueDeclare(queueConfig, self.amqpChannel);
        if (result is handle) {
            return java:toString(result);
        } else {
            return result;
        }
    }

# Declares a non-auto-delete, non-durable exchange with no extra arguments.
# If the arguments are specified, then the exchange is declared accordingly.
# ```ballerina
# rabbitmq:Error? exchangeResult = newChannel->exchangeDeclare({
#                                               exchangeName: "MyExchange",
#                                               exchangeType: rabbitmq:DIRECT_EXCHANGE,
#                                               durable: true,
#                                               autoDelete: true });
# ```
#
# + exchangeConfig - Configurations required to declare an exchange
# + return - A `rabbitmq:Error` if an I/O error is encountered or else `()`
    public remote function exchangeDeclare(ExchangeConfiguration exchangeConfig) returns Error? {
        return nativeExchangeDeclare(exchangeConfig, self.amqpChannel);
    }

# Binds a queue to an exchange with the given binding key.
# ```ballerina
# rabbitmq:Error? bindResult = newChannel.queueBind("MyQueue", "MyExchange", "routing-key");
# ```
#
# + queueName - Name of the queue
# + exchangeName - Name of the exchange
# + bindingKey - Binding key used to bind the queue to the exchange
# + return - A `rabbitmq:Error` if an I/O error is encountered or else `()`
    public remote function queueBind(string queueName, string exchangeName, string bindingKey) returns Error? {
         return nativeQueueBind(java:fromString(queueName), java:fromString(exchangeName),
                    java:fromString(bindingKey), self.amqpChannel);
    }

# Publishes a message. Publishing to a non-existent exchange will result in a channel-level
# protocol error, which closes the channel.
# ```ballerina
# rabbitmq:Error? sendResult = newChannel->basicPublish("Hello from Ballerina", "MyQueue");
# ```
#
# + messageContent - The message body
# + routingKey - The routing key
# + exchangeName - The name of the exchange to which the message is published
# + properties - Other properties for the message (routing headers, etc.)
# + return - A `rabbitmq:Error` if an I/O error is encountered or else `()`
    public remote function basicPublish(@untainted MessageContent messageContent, string routingKey,
                        string exchangeName = "", public BasicProperties? properties = ()) returns Error? {
        return nativeBasicPublish(messageContent, java:fromString(routingKey),
         java:fromString(exchangeName), properties, self.amqpChannel);
    }

# Deletes the queue with the given name although it is in use or has messages in it.
# If the `ifUnused` or `ifEmpty` parameters are given, the queue is checked before deleting.
# ```ballerina
# rabbitmq:Error? deleteResult = newChannel->queueDelete("MyQueue");
# ```
#
# + queueName - Name of the queue to be deleted
# + ifUnused - True if the queue should be deleted only if it's not in use
# + ifEmpty - True if the queue should be deleted only if it's empty
# + return - A `rabbitmq:Error` if an I/O error is encountered or else `()`
    public remote function queueDelete(string queueName, boolean ifUnused = false, boolean ifEmpty = false)
                        returns Error? {
        return nativeQueueDelete(java:fromString(queueName), ifUnused, ifEmpty, self.amqpChannel);
    }

# Deletes the exchange with the given name.
# ```ballerina
# rabbitmq:Error? deleteResult = newChannel->exchangeDelete("MyExchange");
# ```
#
# + exchangeName - The name of the exchange
# + return - A `rabbitmq:Error` if an I/O error is encountered or else `()`
    public remote function exchangeDelete(string exchangeName) returns Error? {
        return nativeExchangeDelete(java:fromString(exchangeName), self.amqpChannel);
    }

# Purges the content of the given queue.
# ```ballerina
# rabbitmq:Error? purgeResult = newChannel->queuePurge("MyQueue");
# ```
#
# + queueName - The name of the queue
# + return - A `rabbitmq:Error` if an I/O error is encountered or else `()`
    public remote function queuePurge(string queueName) returns Error? {
        return nativeQueuePurge(java:fromString(queueName), self.amqpChannel);
    }

# Retrieves a message synchronously from the given queue providing direct access to the messages in the queue.
# ```ballerina
# rabbitmq:Message|rabbitmq:Error getResult = newChannel->basicGet("MyQueue", rabbitmq:AUTO_ACK);
# ```
#
# + queueName - The name of the queue
# + ackMode - Type of the acknowledgement mode
# + return - A `rabbitmq:Message` object containing the retrieved message data or else a`rabbitmq:Error` if an
#            I/O error is encountered
    public remote function basicGet(string queueName, AcknowledgementMode ackMode) returns Message | Error {
        boolean autoAck = ackMode is AUTO_ACK;
        return nativeBasicGet(java:fromString(queueName), autoAck, self.amqpChannel);
    }

# Retrieves the connection, which carries this channel.
# ```ballerina
# rabbitmq:Connection|rabbitmq:Error connResult = newChannel.getConnection();
# ```
#
# + return - A `rabbitmq:Connection` object or else a `rabbitmq:Error` if an I/O error is encountered
    public function getConnection() returns Connection | Error {
        return nativeGetConnection(self.amqpChannel);
    }

# Closes the `rabbitmq:Channel`.
# ```ballerina
# rabbitmq:Error? closeResult = newChannel.close();
# ```
#
# + closeCode - The close code (for information, go to the "Reply Codes" section in the
#               [AMQP 0-9-1 specification] (#https://www.rabbitmq.com/resources/specs/amqp0-9-1.pdf))
# + closeMessage - A message indicating the reason for closing the channel
# + return - A `rabbitmq:Error` if an I/O error is encountered or else `()`
    public function close(int? closeCode = (), string? closeMessage = ()) returns Error? {
        return nativeChannelClose(closeCode, closeMessage, self.amqpChannel);
    }

# Aborts the RabbitMQ `rabbitmq:Channel`. Forces the `rabbitmq:Channel` to close and waits for all the close operations
# to complete. Any encountered exceptions in the close operations are discarded silently.
# ```ballerina
# rabbitmq:Error? abortResult = newChannel.abortChannel(320, "Channel Aborted");
# ```
#
# + closeCode - The close code (for information, go to the "Reply Codes" section in the
#               [AMQP 0-9-1 specification] (#https://www.rabbitmq.com/resources/specs/amqp0-9-1.pdf))
# + closeMessage - A message indicating the reason for closing the channel
# + return - A `rabbitmq:Error` if an I/O error is encountered or else `()`
    public function abortChannel(int? closeCode = (), string? closeMessage = ()) returns Error? {
        return nativeChannelAbort(closeCode, closeMessage, self.amqpChannel);
    }

    function getChannel() returns handle {
        return self.amqpChannel;
    }
};

function createChannel(handle connection) returns handle =
@java:Method {
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeQueueDeclare(QueueConfiguration? config, handle amqpChannel) returns handle | Error? =
@java:Method {
    name: "queueDeclare",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeExchangeDeclare(ExchangeConfiguration config, handle amqpChannel) returns Error? =
@java:Method {
    name: "exchangeDeclare",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeQueueBind(handle queueName, handle exchangeName, handle bindingKey, handle amqpChannel) returns Error? =
@java:Method {
    name: "queueBind",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeBasicPublish(MessageContent messageContent, handle routingKey, handle exchangeName,
BasicProperties? properties, handle amqpChannel) returns Error? =
@java:Method {
    name: "basicPublish",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeQueueDelete(handle queueName, boolean ifUnused, boolean ifEmpty, handle amqpChannel) returns Error? =
@java:Method {
    name: "queueDelete",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeExchangeDelete(handle exchangeName, handle amqpChannel) returns Error? =
@java:Method {
    name: "exchangeDelete",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeQueuePurge(handle queueName, handle amqpChannel) returns Error? =
@java:Method {
    name: "queuePurge",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeChannelClose(int? closeCode, string? closeMessage, handle amqpChannel) returns Error? =
@java:Method {
    name: "close",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeChannelAbort(int? closeCode, string? closeMessage, handle amqpChannel) returns Error? =
@java:Method {
    name: "abort",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeGetConnection(handle amqpChannel) returns Connection | Error =
@java:Method {
    name: "getConnection",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;

function nativeBasicGet(handle queueName, boolean ackMode, handle amqpChannel) returns Message | Error =
@java:Method {
    name: "basicGet",
    class: "org.ballerinalang.messaging.rabbitmq.util.ChannelUtils"
} external;
