/*
 * Decompiled with CFR 0.152.
 */
package io.ballerina.stdlib.crypto;

import io.ballerina.runtime.api.Module;
import io.ballerina.runtime.api.creators.ErrorCreator;
import io.ballerina.runtime.api.creators.ValueCreator;
import io.ballerina.runtime.api.utils.StringUtils;
import io.ballerina.runtime.api.values.BArray;
import io.ballerina.runtime.api.values.BError;
import io.ballerina.runtime.api.values.BString;
import io.ballerina.stdlib.crypto.nativeimpl.ModuleUtils;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.Signature;
import java.security.SignatureException;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Arrays;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.bouncycastle.crypto.DerivationFunction;
import org.bouncycastle.crypto.DerivationParameters;
import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.SecretWithEncapsulation;
import org.bouncycastle.crypto.digests.SHA256Digest;
import org.bouncycastle.crypto.generators.HKDFBytesGenerator;
import org.bouncycastle.crypto.generators.KDF2BytesGenerator;
import org.bouncycastle.crypto.kems.RSAKEMExtractor;
import org.bouncycastle.crypto.kems.RSAKEMGenerator;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.HKDFParameters;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.crypto.params.RSAKeyParameters;
import org.bouncycastle.jcajce.SecretKeyWithEncapsulation;
import org.bouncycastle.jcajce.spec.KEMExtractSpec;
import org.bouncycastle.jcajce.spec.KEMGenerateSpec;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

public class CryptoUtils {
    private static final int[] VALID_GCM_TAG_SIZES = new int[]{32, 63, 96, 104, 112, 120, 128};
    private static final int[] VALID_AES_KEY_SIZES = new int[]{16, 24, 32};

    private CryptoUtils() {
    }

    public static Object hmac(String algorithm, byte[] key, byte[] input) {
        try {
            SecretKeySpec secretKey = new SecretKeySpec(key, algorithm);
            Mac mac = Mac.getInstance(algorithm);
            mac.init(secretKey);
            return ValueCreator.createArrayValue((byte[])mac.doFinal(input));
        }
        catch (IllegalArgumentException | InvalidKeyException e) {
            return CryptoUtils.createError("Error occurred while calculating HMAC: " + e.getMessage());
        }
        catch (NoSuchAlgorithmException e) {
            throw CryptoUtils.createError("Error occurred while calculating HMAC: " + e.getMessage());
        }
    }

    public static byte[] hash(String algorithm, byte[] input, Object salt) {
        try {
            CryptoUtils.addBCProvider();
            MessageDigest messageDigest = MessageDigest.getInstance(algorithm);
            if (salt != null) {
                messageDigest.update(((BArray)salt).getBytes());
            }
            return messageDigest.digest(input);
        }
        catch (NoSuchAlgorithmException e) {
            throw CryptoUtils.createError("Error occurred while calculating hash: " + e.getMessage());
        }
    }

    public static Object sign(String algorithm, PrivateKey privateKey, byte[] input) {
        try {
            Signature sig = Signature.getInstance(algorithm);
            sig.initSign(privateKey);
            sig.update(input);
            return ValueCreator.createArrayValue((byte[])sig.sign());
        }
        catch (InvalidKeyException e) {
            return CryptoUtils.createError("Uninitialized private key: " + e.getMessage());
        }
        catch (SignatureException e) {
            return CryptoUtils.createError("Error occurred while calculating signature: " + e.getMessage());
        }
        catch (NoSuchAlgorithmException e) {
            throw CryptoUtils.createError("Error occurred while calculating signature: " + e.getMessage());
        }
    }

    public static Object verify(String algorithm, PublicKey publicKey, byte[] data, byte[] signature) {
        try {
            Signature sig = Signature.getInstance(algorithm);
            sig.initVerify(publicKey);
            sig.update(data);
            return sig.verify(signature);
        }
        catch (InvalidKeyException e) {
            return CryptoUtils.createError("Uninitialized public key: " + e.getMessage());
        }
        catch (SignatureException e) {
            return CryptoUtils.createError("Error occurred while calculating signature: " + e.getMessage());
        }
        catch (NoSuchAlgorithmException e) {
            throw CryptoUtils.createError("Error occurred while calculating signature: " + e.getMessage());
        }
    }

    public static Object hkdf(String digestAlgorithm, byte[] ikm, byte[] salt, byte[] info, int length) {
        Digest hash = CryptoUtils.selectHash(digestAlgorithm);
        byte[] okm = new byte[length];
        HKDFParameters params = new HKDFParameters(ikm, salt, info);
        HKDFBytesGenerator hkdf = new HKDFBytesGenerator(hash);
        hkdf.init((DerivationParameters)params);
        hkdf.generateBytes(okm, 0, length);
        return ValueCreator.createArrayValue((byte[])okm);
    }

    public static Object generateEncapsulated(String algorithm, PublicKey publicKey, String provider) {
        try {
            KEMGenerateSpec kemGenerateSpec = new KEMGenerateSpec(publicKey, algorithm);
            KeyGenerator keyGenerator = KeyGenerator.getInstance(algorithm, provider);
            keyGenerator.init((AlgorithmParameterSpec)kemGenerateSpec);
            return keyGenerator.generateKey();
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException e) {
            return CryptoUtils.createError("Error occurred while generating encapsulated key: " + e.getMessage());
        }
        catch (NoSuchProviderException e) {
            throw CryptoUtils.createError("Provider not found: " + provider);
        }
    }

    public static Object generateRsaEncapsulated(PublicKey publicKey) {
        if (!(publicKey instanceof RSAPublicKey)) {
            return CryptoUtils.createError("Error occurred while generating encapsulated key: valid RSA public key expected");
        }
        RSAPublicKey rsaPublicKey = (RSAPublicKey)publicKey;
        RSAKEMGenerator keyGenerator = new RSAKEMGenerator(32, (DerivationFunction)new KDF2BytesGenerator((Digest)new SHA256Digest()), new SecureRandom());
        RSAKeyParameters rsaKeyParams = new RSAKeyParameters(false, rsaPublicKey.getModulus(), rsaPublicKey.getPublicExponent());
        SecretWithEncapsulation secretWithEncapsulation = keyGenerator.generateEncapsulated((AsymmetricKeyParameter)rsaKeyParams);
        SecretKeySpec secretKey = new SecretKeySpec(secretWithEncapsulation.getSecret(), "RSA");
        return new SecretKeyWithEncapsulation((SecretKey)secretKey, secretWithEncapsulation.getEncapsulation());
    }

    public static Object extractSecret(byte[] encapsulation, String algorithm, PrivateKey privateKey, String provider) {
        try {
            KEMExtractSpec kemExtractSpec = new KEMExtractSpec(privateKey, encapsulation, algorithm);
            KeyGenerator keyGenerator = KeyGenerator.getInstance(algorithm, provider);
            keyGenerator.init((AlgorithmParameterSpec)kemExtractSpec);
            SecretKeyWithEncapsulation secretKeyWithEncapsulation = (SecretKeyWithEncapsulation)keyGenerator.generateKey();
            return ValueCreator.createArrayValue((byte[])secretKeyWithEncapsulation.getEncoded());
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException e) {
            return CryptoUtils.createError("Error occurred while extracting secret: " + e.getMessage());
        }
        catch (NoSuchProviderException e) {
            throw CryptoUtils.createError("Provider not found: " + e.getMessage());
        }
    }

    public static Object extractRsaSecret(byte[] encapsulation, PrivateKey privateKey) {
        if (!(privateKey instanceof RSAPrivateKey)) {
            return CryptoUtils.createError("Error occurred while extracting secret: valid RSA privatekey expected");
        }
        RSAPrivateKey rsaPrivateKey = (RSAPrivateKey)privateKey;
        RSAKeyParameters rsaKeyParameters = new RSAKeyParameters(true, rsaPrivateKey.getModulus(), rsaPrivateKey.getPrivateExponent());
        RSAKEMExtractor keyExtractor = new RSAKEMExtractor(rsaKeyParameters, 32, (DerivationFunction)new KDF2BytesGenerator((Digest)new SHA256Digest()));
        KeyParameter keyParameter = new KeyParameter(keyExtractor.extractSecret(encapsulation));
        return ValueCreator.createArrayValue((byte[])keyParameter.getKey());
    }

    public static BError createError(String errMsg) {
        return ErrorCreator.createDistinctError((String)"Error", (Module)ModuleUtils.getModule(), (BString)StringUtils.fromString((String)errMsg));
    }

    public static Object rsaEncryptDecrypt(CipherMode cipherMode, String algorithmMode, String algorithmPadding, Key key, byte[] input, byte[] iv, long tagSize) {
        try {
            String transformedAlgorithmPadding = CryptoUtils.transformAlgorithmPadding(algorithmPadding);
            if (tagSize != -1L && Arrays.stream(VALID_GCM_TAG_SIZES).noneMatch(i -> tagSize == (long)i)) {
                return CryptoUtils.createError("Valid tag sizes are: " + Arrays.toString(VALID_GCM_TAG_SIZES));
            }
            AlgorithmParameterSpec paramSpec = CryptoUtils.buildParameterSpec(algorithmMode, iv, (int)tagSize);
            Cipher cipher = Cipher.getInstance("RSA/" + algorithmMode + "/" + transformedAlgorithmPadding);
            CryptoUtils.initCipher(cipher, cipherMode, key, paramSpec);
            return ValueCreator.createArrayValue((byte[])cipher.doFinal(input));
        }
        catch (NoSuchAlgorithmException e) {
            return CryptoUtils.createError("Unsupported algorithm: RSA " + algorithmMode + " " + algorithmPadding + ": " + e.getMessage());
        }
        catch (NoSuchPaddingException e) {
            return CryptoUtils.createError("Unsupported padding scheme defined in the algorithm: RSA " + algorithmMode + " " + algorithmPadding + ": " + e.getMessage());
        }
        catch (BError | InvalidAlgorithmParameterException | InvalidKeyException | BadPaddingException | IllegalBlockSizeException e) {
            return CryptoUtils.createError("Error occurred while RSA encrypt/decrypt: " + e.getMessage());
        }
    }

    public static Object aesEncryptDecrypt(CipherMode cipherMode, String algorithmMode, String algorithmPadding, byte[] key, byte[] input, byte[] iv, long tagSize) {
        try {
            if (Arrays.stream(VALID_AES_KEY_SIZES).noneMatch(validSize -> validSize == key.length)) {
                return CryptoUtils.createError("Invalid key size. Valid key sizes in bytes: " + Arrays.toString(VALID_AES_KEY_SIZES));
            }
            String transformedAlgorithmPadding = CryptoUtils.transformAlgorithmPadding(algorithmPadding);
            SecretKeySpec keySpec = new SecretKeySpec(key, "AES");
            if (tagSize != -1L && Arrays.stream(VALID_GCM_TAG_SIZES).noneMatch(validSize -> (long)validSize == tagSize)) {
                return CryptoUtils.createError("Invalid tag size. Valid tag sizes in bytes: " + Arrays.toString(VALID_GCM_TAG_SIZES));
            }
            AlgorithmParameterSpec paramSpec = CryptoUtils.buildParameterSpec(algorithmMode, iv, (int)tagSize);
            Cipher cipher = Cipher.getInstance("AES/" + algorithmMode + "/" + transformedAlgorithmPadding);
            CryptoUtils.initCipher(cipher, cipherMode, keySpec, paramSpec);
            return ValueCreator.createArrayValue((byte[])cipher.doFinal(input));
        }
        catch (NoSuchAlgorithmException e) {
            return CryptoUtils.createError("Unsupported algorithm: AES " + algorithmMode + " " + algorithmPadding + ": " + e.getMessage());
        }
        catch (NoSuchPaddingException e) {
            return CryptoUtils.createError("Unsupported padding scheme defined in  the algorithm: AES " + algorithmMode + " " + algorithmPadding + ": " + e.getMessage());
        }
        catch (BError | InvalidAlgorithmParameterException | InvalidKeyException | BadPaddingException | IllegalBlockSizeException e) {
            return CryptoUtils.createError("Error occurred while AES encrypt/decrypt: " + e.getMessage());
        }
    }

    public static void addBCProvider() {
        if (Security.getProvider("BC") == null) {
            Security.addProvider((Provider)new BouncyCastleProvider());
        }
    }

    private static void initCipher(Cipher cipher, CipherMode cipherMode, Key key, AlgorithmParameterSpec paramSpec) throws InvalidKeyException, InvalidAlgorithmParameterException {
        switch (cipherMode.ordinal()) {
            case 0: {
                if (paramSpec == null) {
                    cipher.init(1, key);
                    break;
                }
                cipher.init(1, key, paramSpec);
                break;
            }
            case 1: {
                if (paramSpec == null) {
                    cipher.init(2, key);
                    break;
                }
                cipher.init(2, key, paramSpec);
            }
        }
    }

    private static AlgorithmParameterSpec buildParameterSpec(String algorithmMode, byte[] iv, int tagSize) {
        switch (algorithmMode) {
            case "GCM": {
                return new GCMParameterSpec(tagSize, iv);
            }
            case "CBC": {
                return new IvParameterSpec(iv);
            }
        }
        return null;
    }

    private static String transformAlgorithmPadding(String algorithmPadding) throws BError {
        switch (algorithmPadding) {
            case "PKCS1": {
                algorithmPadding = "PKCS1Padding";
                break;
            }
            case "PKCS5": {
                algorithmPadding = "PKCS5Padding";
                break;
            }
            case "OAEPwithMD5andMGF1": {
                algorithmPadding = "OAEPWithMD5AndMGF1Padding";
                break;
            }
            case "OAEPWithSHA1AndMGF1": {
                algorithmPadding = "OAEPWithSHA-1AndMGF1Padding";
                break;
            }
            case "OAEPWithSHA256AndMGF1": {
                algorithmPadding = "OAEPWithSHA-256AndMGF1Padding";
                break;
            }
            case "OAEPwithSHA384andMGF1": {
                algorithmPadding = "OAEPWithSHA-384AndMGF1Padding";
                break;
            }
            case "OAEPwithSHA512andMGF1": {
                algorithmPadding = "OAEPWithSHA-512AndMGF1Padding";
                break;
            }
            case "NONE": {
                algorithmPadding = "NoPadding";
                break;
            }
            default: {
                throw CryptoUtils.createError("Unsupported padding: " + algorithmPadding);
            }
        }
        return algorithmPadding;
    }

    private static Digest selectHash(String algorithm) {
        if ("SHA-256".equals(algorithm)) {
            return new SHA256Digest();
        }
        throw CryptoUtils.createError("Unsupported algorithm: " + algorithm);
    }

    public static enum CipherMode {
        ENCRYPT,
        DECRYPT;

    }
}

