/*
 * Decompiled with CFR 0.152.
 */
package io.ballerina.projects.internal;

import io.ballerina.projects.DependencyGraph;
import io.ballerina.projects.DependencyResolutionType;
import io.ballerina.projects.DiagnosticResult;
import io.ballerina.projects.PackageDependencyScope;
import io.ballerina.projects.PackageDescriptor;
import io.ballerina.projects.PackageVersion;
import io.ballerina.projects.SemanticVersion;
import io.ballerina.projects.environment.ModuleLoadRequest;
import io.ballerina.projects.environment.PackageLockingMode;
import io.ballerina.projects.environment.PackageMetadataResponse;
import io.ballerina.projects.environment.PackageResolver;
import io.ballerina.projects.environment.ResolutionOptions;
import io.ballerina.projects.environment.ResolutionRequest;
import io.ballerina.projects.environment.ResolutionResponse;
import io.ballerina.projects.internal.BlendedManifest;
import io.ballerina.projects.internal.DefaultDiagnosticResult;
import io.ballerina.projects.internal.DotGraphs;
import io.ballerina.projects.internal.ModuleResolver;
import io.ballerina.projects.internal.PackageContainer;
import io.ballerina.projects.internal.PackageDependencyGraphBuilder;
import io.ballerina.projects.internal.PackageDiagnostic;
import io.ballerina.projects.internal.ProjectDiagnosticErrorCode;
import io.ballerina.tools.diagnostics.Diagnostic;
import io.ballerina.tools.diagnostics.DiagnosticInfo;
import io.ballerina.tools.diagnostics.DiagnosticSeverity;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

public class ResolutionEngine {
    private final PackageDescriptor rootPkgDesc;
    private final BlendedManifest blendedManifest;
    private final PackageResolver packageResolver;
    private final ModuleResolver moduleResolver;
    private final ResolutionOptions resolutionOptions;
    private final PackageDependencyGraphBuilder graphBuilder;
    private final List<Diagnostic> diagnostics;
    private String dependencyGraphDump;
    private DiagnosticResult diagnosticResult;
    private Set<DependencyNode> unresolvedDeps = null;

    public ResolutionEngine(PackageDescriptor rootPkgDesc, BlendedManifest blendedManifest, PackageResolver packageResolver, ModuleResolver moduleResolver, ResolutionOptions resolutionOptions) {
        this.rootPkgDesc = rootPkgDesc;
        this.blendedManifest = blendedManifest;
        this.packageResolver = packageResolver;
        this.moduleResolver = moduleResolver;
        this.resolutionOptions = resolutionOptions;
        this.graphBuilder = new PackageDependencyGraphBuilder(rootPkgDesc, resolutionOptions);
        this.diagnostics = new ArrayList<Diagnostic>();
        this.dependencyGraphDump = "";
    }

    public DiagnosticResult diagnosticResult() {
        if (this.diagnosticResult == null) {
            this.diagnosticResult = new DefaultDiagnosticResult(this.diagnostics);
        }
        return this.diagnosticResult;
    }

    public DependencyGraph<DependencyNode> resolveDependencies(Collection<ModuleLoadRequest> moduleLoadRequests) {
        Collection<DependencyNode> directDependencies = this.resolvePackages(moduleLoadRequests);
        this.populateStaticDependencyGraph(directDependencies);
        this.updateDependencyVersions();
        this.completeDependencyGraph();
        return this.buildFinalDependencyGraph();
    }

    private Collection<DependencyNode> resolvePackages(Collection<ModuleLoadRequest> moduleLoadRequests) {
        PackageContainer<ModuleResolver.DirectPackageDependency> directDepsContainer = this.moduleResolver.resolveModuleLoadRequests(moduleLoadRequests);
        ArrayList<DependencyNode> directDeps = new ArrayList<DependencyNode>();
        for (ModuleResolver.DirectPackageDependency directPkgDependency : directDepsContainer.getAll()) {
            String repository;
            PackageVersion depVersion;
            boolean errorNode = false;
            PackageDescriptor depPkgDesc = directPkgDependency.pkgDesc();
            if (directPkgDependency.dependencyKind() == ModuleResolver.DirectPackageDependencyKind.NEW) {
                Optional<BlendedManifest.Dependency> blendedDepOptional = this.blendedManifest.dependency(depPkgDesc.org(), depPkgDesc.name());
                if (blendedDepOptional.isPresent()) {
                    errorNode = blendedDepOptional.get().isError();
                }
                if (directPkgDependency.pkgDesc().version() != null) {
                    depVersion = directPkgDependency.pkgDesc().version();
                    repository = blendedDepOptional.map(BlendedManifest.Dependency::repository).orElse(null);
                } else if (blendedDepOptional.isPresent()) {
                    BlendedManifest.Dependency blendedDep = blendedDepOptional.get();
                    repository = blendedDep.repository();
                    depVersion = blendedDep.version();
                } else {
                    depVersion = null;
                    repository = null;
                }
            } else if (directPkgDependency.dependencyKind() == ModuleResolver.DirectPackageDependencyKind.EXISTING) {
                BlendedManifest.Dependency blendedDep = this.blendedManifest.dependencyOrThrow(depPkgDesc.org(), depPkgDesc.name());
                depVersion = blendedDep.version();
                repository = blendedDep.repository();
                errorNode = blendedDep.isError();
            } else {
                throw new IllegalStateException("Unsupported direct dependency kind: " + String.valueOf((Object)directPkgDependency.dependencyKind()));
            }
            directDeps.add(new DependencyNode(PackageDescriptor.from(depPkgDesc.org(), depPkgDesc.name(), depVersion, repository), directPkgDependency.scope(), directPkgDependency.resolutionType(), errorNode));
        }
        return directDeps;
    }

    private void populateStaticDependencyGraph(Collection<DependencyNode> directDependencies) {
        List<DependencyNode> errorNodes = directDependencies.stream().filter(DependencyNode::errorNode).toList();
        for (DependencyNode errorNode : errorNodes) {
            this.graphBuilder.addErroneousDependency(this.rootPkgDesc, errorNode.pkgDesc, errorNode.scope, errorNode.resolutionType);
        }
        directDependencies.removeAll(errorNodes);
        Collection<PackageMetadataResponse> pkgMetadataResponses = this.resolveDirectDependencies(directDependencies);
        this.unresolvedDeps = new HashSet<DependencyNode>();
        HashSet<DependencyNode> resolvedDeps = new HashSet<DependencyNode>();
        for (PackageMetadataResponse resolutionResp : pkgMetadataResponses) {
            if (resolutionResp.resolutionStatus() == ResolutionResponse.ResolutionStatus.UNRESOLVED) {
                if (!this.resolutionOptions.dumpRawGraphs() && !this.resolutionOptions.dumpGraph()) continue;
                ResolutionRequest resolutionRequest = resolutionResp.packageLoadRequest();
                DependencyNode dependencyNode = new DependencyNode(resolutionRequest.packageDescriptor(), resolutionRequest.scope(), resolutionRequest.resolutionType());
                this.unresolvedDeps.add(dependencyNode);
                this.graphBuilder.addUnresolvedDirectDepToRawGraph(dependencyNode);
                continue;
            }
            ResolutionRequest resolutionReq = resolutionResp.packageLoadRequest();
            PackageDescriptor resolvedPkgDesc = resolutionResp.resolvedDescriptor();
            DependencyResolutionType resolutionType = resolutionReq.resolutionType();
            PackageDependencyScope scope = resolutionReq.scope();
            PackageDependencyGraphBuilder.NodeStatus nodeStatus = this.graphBuilder.addResolvedDependency(this.rootPkgDesc, resolvedPkgDesc, scope, resolutionType);
            if (nodeStatus == PackageDependencyGraphBuilder.NodeStatus.ACCEPTED) {
                this.mergeGraph(resolvedPkgDesc, resolutionResp.dependencyGraph().orElseThrow(() -> new IllegalStateException("Graph cannot be null in the resolved dependency: " + resolvedPkgDesc.toString())), scope, resolutionType);
            }
            resolvedDeps.add(new DependencyNode(resolvedPkgDesc, scope, resolutionType));
        }
        if (this.resolutionOptions.dumpRawGraphs() || this.resolutionOptions.dumpGraph()) {
            HashSet<DependencyNode> unresolvedNodes = new HashSet<DependencyNode>(this.graphBuilder.getAllDependencies());
            unresolvedNodes.removeAll(resolvedDeps);
            this.unresolvedDeps.addAll(unresolvedNodes);
        }
        this.dumpInitialGraph();
    }

    private Collection<PackageMetadataResponse> resolveDirectDependencies(Collection<DependencyNode> directDeps) {
        PackageLockingMode defaultLockingMode = this.resolutionOptions.sticky() ? PackageLockingMode.HARD : this.resolutionOptions.packageLockingMode();
        ArrayList<ResolutionRequest> resolutionRequests = new ArrayList<ResolutionRequest>();
        for (DependencyNode directDependency : directDeps) {
            PackageLockingMode lockingMode = defaultLockingMode;
            PackageDescriptor pkgDesc = directDependency.pkgDesc();
            Optional<BlendedManifest.Dependency> dependency = this.blendedManifest.lockedDependency(pkgDesc.org(), pkgDesc.name());
            if (dependency.isPresent()) {
                if (dependency.get().relation() == BlendedManifest.DependencyRelation.TRANSITIVE) {
                    lockingMode = PackageLockingMode.SOFT;
                }
            } else {
                dependency = this.blendedManifest.userSpecifiedDependency(pkgDesc.org(), pkgDesc.name());
                if (dependency.isPresent() && (dependency.get().isFromLocalRepository() || dependency.get().isFromCustomRepository())) {
                    lockingMode = PackageLockingMode.HARD;
                }
            }
            resolutionRequests.add(ResolutionRequest.from(pkgDesc, directDependency.scope(), directDependency.resolutionType(), lockingMode));
        }
        return this.packageResolver.resolvePackageMetadata(resolutionRequests, this.resolutionOptions);
    }

    private void mergeGraph(PackageDescriptor rootNode, DependencyGraph<PackageDescriptor> dependencyGraph, PackageDependencyScope scope, DependencyResolutionType resolutionType) {
        Collection<PackageDescriptor> directDependencies = dependencyGraph.getDirectDependencies(rootNode);
        for (PackageDescriptor directDep : directDependencies) {
            PackageDependencyGraphBuilder.NodeStatus nodeStatus;
            DependencyGraph<PackageDescriptor> dependencyGraphFinal;
            if (directDep.isBuiltInPackage()) {
                dependencyGraphFinal = this.getBuiltInPkgDescDepGraph(scope, directDep);
                nodeStatus = this.graphBuilder.addResolvedDependency(rootNode, directDep, scope, resolutionType);
            } else {
                dependencyGraphFinal = dependencyGraph;
                nodeStatus = this.graphBuilder.addUnresolvedDependency(rootNode, directDep, scope, resolutionType);
            }
            if (nodeStatus != PackageDependencyGraphBuilder.NodeStatus.ACCEPTED) continue;
            this.mergeGraph(directDep, dependencyGraphFinal, scope, resolutionType);
        }
    }

    private DependencyGraph<PackageDescriptor> getBuiltInPkgDescDepGraph(PackageDependencyScope scope, PackageDescriptor directDep) {
        Collection<PackageMetadataResponse> packageMetadataResponses = this.packageResolver.resolvePackageMetadata(Collections.singletonList(ResolutionRequest.from(directDep, scope)), this.resolutionOptions);
        if (packageMetadataResponses.isEmpty()) {
            throw new IllegalStateException("built-in package not found in distribution: " + directDep.toString());
        }
        PackageMetadataResponse packageMetadataResponse = packageMetadataResponses.iterator().next();
        Optional<DependencyGraph<PackageDescriptor>> packageDescriptorDependencyGraph = packageMetadataResponse.dependencyGraph();
        return packageDescriptorDependencyGraph.orElseThrow(() -> new IllegalStateException("Graph cannot be null in the built-in package: " + directDep.toString()));
    }

    private void updateDependencyVersions() {
        this.graphBuilder.removeDanglingNodes();
        Collection<DependencyNode> unresolvedNodes = this.getUnresolvedNode();
        ArrayList<DependencyNode> errorNodes = new ArrayList<DependencyNode>();
        ArrayList<ResolutionRequest> unresolvedRequests = new ArrayList<ResolutionRequest>();
        for (DependencyNode unresolvedNode : unresolvedNodes) {
            if (unresolvedNode.isError) {
                errorNodes.add(unresolvedNode);
                continue;
            }
            PackageDescriptor unresolvedPkgDes = unresolvedNode.pkgDesc();
            Optional<BlendedManifest.Dependency> blendedDepOptional = this.blendedManifest.dependency(unresolvedPkgDes.org(), unresolvedPkgDes.name());
            ResolutionRequest resolutionRequest = this.getRequestForUnresolvedNode(unresolvedNode, blendedDepOptional.orElse(null));
            if (resolutionRequest == null) {
                errorNodes.add(new DependencyNode(unresolvedNode.pkgDesc, unresolvedNode.scope, unresolvedNode.resolutionType, true));
                continue;
            }
            unresolvedRequests.add(resolutionRequest);
        }
        Collection<PackageMetadataResponse> pkgMetadataResponses = this.packageResolver.resolvePackageMetadata(unresolvedRequests, this.resolutionOptions);
        this.addUpdatedPackagesToGraph(pkgMetadataResponses);
        this.addErrorNodesToGraph(errorNodes);
        this.dumpIntermediateGraph(1);
    }

    private void addErrorNodesToGraph(List<DependencyNode> errorNodes) {
        for (DependencyNode errorNode : errorNodes) {
            this.graphBuilder.addErrorNode(errorNode.pkgDesc, errorNode.scope, errorNode.resolutionType);
            if (!this.resolutionOptions.dumpGraph() && !this.resolutionOptions.dumpRawGraphs()) continue;
            this.unresolvedDeps.remove(errorNode);
        }
    }

    private ResolutionRequest getRequestForUnresolvedNode(DependencyNode unresolvedNode, BlendedManifest.Dependency blendedDep) {
        if (blendedDep == null) {
            return ResolutionRequest.from(unresolvedNode.pkgDesc(), unresolvedNode.scope(), unresolvedNode.resolutionType(), this.resolutionOptions.packageLockingMode());
        }
        if (blendedDep.isError()) {
            return null;
        }
        SemanticVersion.VersionCompatibilityResult versionCompResult = blendedDep.version().compareTo(unresolvedNode.pkgDesc().version());
        if (versionCompResult == SemanticVersion.VersionCompatibilityResult.GREATER_THAN || versionCompResult == SemanticVersion.VersionCompatibilityResult.EQUAL) {
            PackageLockingMode lockingMode = this.resolutionOptions.sticky() || blendedDep.isFromLocalRepository() ? PackageLockingMode.HARD : this.resolutionOptions.packageLockingMode();
            PackageDescriptor blendedDepPkgDesc = PackageDescriptor.from(blendedDep.org(), blendedDep.name(), blendedDep.version(), blendedDep.repository());
            return ResolutionRequest.from(blendedDepPkgDesc, unresolvedNode.scope(), unresolvedNode.resolutionType(), lockingMode);
        }
        if (versionCompResult == SemanticVersion.VersionCompatibilityResult.LESS_THAN) {
            return ResolutionRequest.from(unresolvedNode.pkgDesc(), unresolvedNode.scope(), unresolvedNode.resolutionType(), this.resolutionOptions.packageLockingMode());
        }
        String depInfo = String.valueOf(blendedDep.org()) + "/" + String.valueOf(blendedDep.name());
        String sourceFile = blendedDep.origin() == BlendedManifest.DependencyOrigin.USER_SPECIFIED ? "Ballerina.toml" : "Dependencies.toml";
        DiagnosticInfo diagnosticInfo = new DiagnosticInfo(ProjectDiagnosticErrorCode.INCOMPATIBLE_DEPENDENCY_VERSIONS.diagnosticId(), "Incompatible versions: " + depInfo + ". Version specified in " + sourceFile + ": " + String.valueOf(blendedDep.version()) + " and the version resolved from other dependencies: " + String.valueOf(unresolvedNode.pkgDesc.version()), DiagnosticSeverity.ERROR);
        PackageDiagnostic diagnostic = new PackageDiagnostic(diagnosticInfo, this.rootPkgDesc.name().toString());
        this.diagnostics.add(diagnostic);
        return null;
    }

    private Collection<DependencyNode> getUnresolvedNode() {
        if (this.resolutionOptions.sticky()) {
            return this.graphBuilder.getUnresolvedNodes();
        }
        return this.graphBuilder.getAllDependencies();
    }

    private void completeDependencyGraph() {
        Collection<DependencyNode> unresolvedNodes;
        int noOfUpdateAttempts = 2;
        this.graphBuilder.removeDanglingNodes();
        ArrayList<PackageMetadataResponse> pkgMetadataResponses = new ArrayList<PackageMetadataResponse>();
        while (!(unresolvedNodes = this.graphBuilder.getUnresolvedNodes()).isEmpty()) {
            ArrayList<ResolutionRequest> unresolvedRequests = new ArrayList<ResolutionRequest>(unresolvedNodes.size());
            for (DependencyNode unresolvedNode : unresolvedNodes) {
                PackageDescriptor unresolvedPkgDes = unresolvedNode.pkgDesc();
                Optional<BlendedManifest.Dependency> blendedDepOptional = this.blendedManifest.userSpecifiedDependency(unresolvedPkgDes.org(), unresolvedPkgDes.name());
                ResolutionRequest resolutionRequest = this.getRequestForUnresolvedNode(unresolvedNode, blendedDepOptional.orElse(null));
                if (unresolvedNode.errorNode()) {
                    pkgMetadataResponses.add(PackageMetadataResponse.createUnresolvedResponse(resolutionRequest));
                    continue;
                }
                unresolvedRequests.add(resolutionRequest);
            }
            pkgMetadataResponses.addAll(this.packageResolver.resolvePackageMetadata(unresolvedRequests, this.resolutionOptions));
            this.addUpdatedPackagesToGraph(pkgMetadataResponses);
            this.graphBuilder.removeDanglingNodes();
            this.dumpIntermediateGraph(noOfUpdateAttempts++);
        }
    }

    private void addUpdatedPackagesToGraph(Collection<PackageMetadataResponse> pkgMetadataResponses) {
        for (PackageMetadataResponse resolutionResp : pkgMetadataResponses) {
            if (resolutionResp.resolutionStatus() == ResolutionResponse.ResolutionStatus.UNRESOLVED) continue;
            this.addNodeToGraph(resolutionResp);
        }
    }

    private void addNodeToGraph(PackageMetadataResponse resolutionResp) {
        DependencyResolutionType resolvedType;
        PackageDependencyScope scope;
        ResolutionRequest resolutionReq = resolutionResp.packageLoadRequest();
        PackageDescriptor pkgDesc = resolutionResp.resolvedDescriptor();
        PackageDependencyGraphBuilder.NodeStatus nodeStatus = this.graphBuilder.addResolvedNode(pkgDesc, scope = resolutionReq.scope(), resolvedType = resolutionReq.resolutionType());
        if (nodeStatus == PackageDependencyGraphBuilder.NodeStatus.ACCEPTED) {
            this.mergeGraph(pkgDesc, resolutionResp.dependencyGraph().orElseThrow(() -> new IllegalStateException("Graph cannot be null in the resolved dependency: " + pkgDesc.toString())), scope, resolvedType);
        }
        if (this.resolutionOptions.dumpGraph() || this.resolutionOptions.dumpRawGraphs()) {
            this.unresolvedDeps.remove(new DependencyNode(pkgDesc, scope, resolvedType));
        }
    }

    private DependencyGraph<DependencyNode> buildFinalDependencyGraph() {
        DependencyGraph<DependencyNode> dependencyGraph = this.graphBuilder.buildGraph();
        this.diagnostics.addAll(this.graphBuilder.diagnostics());
        this.dumpFinalGraph(dependencyGraph);
        return dependencyGraph;
    }

    private void dumpInitialGraph() {
        if (!this.resolutionOptions.dumpRawGraphs()) {
            return;
        }
        String serializedGraph = this.serializeRawGraph("Initial");
        this.dependencyGraphDump = this.dependencyGraphDump + "\n";
        this.dependencyGraphDump = this.dependencyGraphDump + serializedGraph + "\n";
    }

    private void dumpIntermediateGraph(int noOfUpdateAttempts) {
        if (!this.resolutionOptions.dumpRawGraphs()) {
            return;
        }
        String serializedGraph = this.serializeRawGraph("Version update attempt " + noOfUpdateAttempts);
        this.dependencyGraphDump = this.dependencyGraphDump + "\n";
        this.dependencyGraphDump = this.dependencyGraphDump + serializedGraph + "\n";
    }

    private void dumpFinalGraph(DependencyGraph<DependencyNode> dependencyGraph) {
        if (!this.resolutionOptions.dumpGraph() && !this.resolutionOptions.dumpRawGraphs()) {
            return;
        }
        ArrayList<DependencyNode> unresolvedDirectDeps = new ArrayList<DependencyNode>(this.graphBuilder.rawGraph().getDirectDependencies(dependencyGraph.getRoot()));
        Collection<DependencyNode> resolvedDirectDeps = dependencyGraph.getDirectDependencies(dependencyGraph.getRoot());
        unresolvedDirectDeps.removeAll(resolvedDirectDeps);
        String serializedGraph = this.resolutionOptions.dumpRawGraphs() ? DotGraphs.serializeDependencyNodeGraph(dependencyGraph, "Final", this.unresolvedDeps, unresolvedDirectDeps) : DotGraphs.serializeDependencyNodeGraph(dependencyGraph, this.unresolvedDeps, unresolvedDirectDeps);
        this.dependencyGraphDump = this.dependencyGraphDump + "\n";
        this.dependencyGraphDump = this.dependencyGraphDump + serializedGraph + "\n";
    }

    private String serializeRawGraph(String graphName) {
        DependencyGraph<DependencyNode> initialGraph = this.graphBuilder.rawGraph();
        return DotGraphs.serializeDependencyNodeGraph(initialGraph, graphName, this.unresolvedDeps);
    }

    public String dumpGraphs() {
        return this.dependencyGraphDump;
    }

    public static class DependencyNode
    implements Comparable<DependencyNode> {
        private final PackageDescriptor pkgDesc;
        private final PackageDependencyScope scope;
        private final DependencyResolutionType resolutionType;
        private final boolean isError;

        public DependencyNode(PackageDescriptor pkgDesc, PackageDependencyScope scope, DependencyResolutionType resolutionType) {
            this.pkgDesc = Objects.requireNonNull(pkgDesc);
            this.scope = Objects.requireNonNull(scope);
            this.resolutionType = Objects.requireNonNull(resolutionType);
            this.isError = false;
        }

        public DependencyNode(PackageDescriptor pkgDesc, PackageDependencyScope scope, DependencyResolutionType resolutionType, boolean errorNode) {
            this.pkgDesc = Objects.requireNonNull(pkgDesc);
            this.scope = Objects.requireNonNull(scope);
            this.resolutionType = Objects.requireNonNull(resolutionType);
            this.isError = errorNode;
        }

        public DependencyNode(PackageDescriptor pkgDesc) {
            this(pkgDesc, PackageDependencyScope.DEFAULT, DependencyResolutionType.SOURCE);
        }

        public PackageDescriptor pkgDesc() {
            return this.pkgDesc;
        }

        public PackageDependencyScope scope() {
            return this.scope;
        }

        public DependencyResolutionType resolutionType() {
            return this.resolutionType;
        }

        public boolean errorNode() {
            return this.isError;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            DependencyNode that = (DependencyNode)o;
            return Objects.equals(this.pkgDesc.org(), that.pkgDesc.org()) && Objects.equals(this.pkgDesc.name(), that.pkgDesc.name()) && Objects.equals(this.pkgDesc.version(), that.pkgDesc.version()) && Objects.equals(this.pkgDesc.repository(), that.pkgDesc.repository()) && this.scope == that.scope && this.resolutionType == that.resolutionType && this.isError == that.isError;
        }

        public int hashCode() {
            return Objects.hash(new Object[]{this.pkgDesc, this.scope, this.resolutionType});
        }

        public String toString() {
            String attr = " [scope=" + String.valueOf((Object)this.scope) + ",kind=" + String.valueOf((Object)this.resolutionType) + ",repo=" + (String)this.pkgDesc.repository().orElse(null) + ",error=" + this.isError + "]";
            return this.pkgDesc.toString() + attr;
        }

        @Override
        public int compareTo(DependencyNode other) {
            return this.pkgDesc.toString().compareTo(other.pkgDesc.toString());
        }
    }
}

