/*
 * Decompiled with CFR 0.152.
 */
package io.ballerina.persist.cmd;

import io.ballerina.cli.BLauncherCmd;
import io.ballerina.persist.BalException;
import io.ballerina.persist.cmd.Utils;
import io.ballerina.persist.models.Module;
import io.ballerina.persist.nodegenerator.SourceGenerator;
import io.ballerina.persist.nodegenerator.syntax.constants.BalSyntaxConstants;
import io.ballerina.persist.nodegenerator.syntax.utils.TomlSyntaxUtils;
import io.ballerina.persist.utils.BalProjectUtils;
import io.ballerina.projects.util.ProjectUtils;
import io.ballerina.toml.syntax.tree.DocumentMemberDeclarationNode;
import io.ballerina.toml.syntax.tree.DocumentNode;
import io.ballerina.toml.syntax.tree.NodeList;
import io.ballerina.toml.syntax.tree.SyntaxTree;
import io.ballerina.toml.syntax.tree.TableNode;
import io.ballerina.tools.text.TextDocument;
import io.ballerina.tools.text.TextDocuments;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Objects;
import picocli.CommandLine;

@CommandLine.Command(name="generate", description={"Generate Ballerina client object for the entity."})
public class Generate
implements BLauncherCmd {
    private static final PrintStream errStream = System.err;
    private final String sourcePath;
    private static final String COMMAND_IDENTIFIER = "persist-generate";
    @CommandLine.Option(names={"-h", "--help"}, hidden=true)
    private boolean helpFlag;
    @CommandLine.Option(names={"--module"})
    private String module;
    @CommandLine.Option(names={"--datastore"})
    private String datastore;
    @CommandLine.Option(names={"--test-datastore"}, description={"Test data store for the generated Ballerina client"})
    private String testDatastore;

    public Generate() {
        this("");
    }

    public Generate(String sourcePath) {
        this.sourcePath = sourcePath;
    }

    public void execute() {
        Module entityModule;
        Path schemaFilePath;
        boolean hasPersistConfig;
        Path generatedSourceDirPath;
        String packageName;
        if (this.helpFlag) {
            String commandUsageInfo = BLauncherCmd.getCommandUsageInfo((String)COMMAND_IDENTIFIER);
            errStream.println(commandUsageInfo);
            return;
        }
        if (Objects.isNull(this.datastore) || this.datastore.isBlank()) {
            errStream.println("ERROR: datastore is required");
            return;
        }
        try {
            BalProjectUtils.validateDatastore(this.datastore);
        }
        catch (BalException e) {
            errStream.printf("ERROR: %s%n", e.getMessage());
            return;
        }
        try {
            BalProjectUtils.validateTestDatastore(this.datastore, this.testDatastore);
        }
        catch (BalException e) {
            errStream.printf("ERROR: %s%n", e.getMessage());
            return;
        }
        Path projectPath = Paths.get(this.sourcePath, new String[0]);
        try {
            BalProjectUtils.validateBallerinaProject(projectPath);
        }
        catch (BalException e) {
            errStream.println(e.getMessage());
            return;
        }
        try {
            packageName = TomlSyntaxUtils.readPackageName(this.sourcePath);
        }
        catch (BalException e) {
            errStream.println(e.getMessage());
            return;
        }
        if (this.module == null) {
            generatedSourceDirPath = Paths.get(this.sourcePath, new String[0]);
            this.module = packageName;
        } else {
            generatedSourceDirPath = Paths.get(this.sourcePath, "modules");
            this.module = this.module.replaceAll("\"", "");
        }
        String moduleNameWithPackage = packageName.equals(this.module) ? packageName : packageName + "." + this.module;
        try {
            hasPersistConfig = Generate.hasPersistConfig(Paths.get(this.sourcePath, "Ballerina.toml"));
        }
        catch (BalException e) {
            errStream.println("ERROR: The project does not contain a toml file");
            return;
        }
        if (hasPersistConfig) {
            errStream.println("The behavior of the `bal persist generate` command has been updated starting from Ballerina update 09.");
            errStream.println("You now have the following options for code generation:");
            errStream.println(System.lineSeparator() + "- `bal persist add --datastore <datastore> --module <module>`: This command adds an entry to \"Ballerina.toml\" to integrate code generation with the package build process.");
            errStream.println("- `bal persist generate --datastore <datastore> --module <module>`: This command performs a one-time generation of the client.");
            errStream.println(System.lineSeparator() + "If you choose to proceed with the `bal persist generate` command, please follow these steps:");
            errStream.println(System.lineSeparator() + "1. Remove the [persist] configuration in the \"Ballerina.toml\" file.");
            errStream.println("2. Delete any previously generated source files, if they exist.");
            errStream.println("3. Re-execute the `bal persist generate --datastore <datastore> --module <module>` command.");
            errStream.println(System.lineSeparator() + "If you have any questions or need further assistance, refer to the updated documentation.");
            return;
        }
        if (!ProjectUtils.validateModuleName((String)moduleNameWithPackage)) {
            errStream.println("ERROR: invalid module name : '" + this.module + "' :\nmodule name can only contain alphanumerics, underscores and periods");
            return;
        }
        if (!ProjectUtils.validateNameLength((String)moduleNameWithPackage)) {
            errStream.println("ERROR: invalid module name : '" + this.module + "' :\nmaximum length of module name is 256 characters");
            return;
        }
        if (!this.module.equals(packageName)) {
            generatedSourceDirPath = generatedSourceDirPath.resolve(this.module);
        }
        if (Files.isDirectory(Paths.get(this.sourcePath, "tool.persist", "migrations"), new LinkOption[0]) && !this.datastore.equals("mysql")) {
            errStream.println("ERROR: regenerating the client with a different datastore after executing the migrate command is not permitted. please remove the migrations directory within the persist directory and try executing the command again.");
            return;
        }
        if (this.datastore.equals("googlesheets")) {
            errStream.printf(BalSyntaxConstants.EXPERIMENTAL_NOTICE, "The support for Google Sheets data store is currently an experimental feature, and its behavior may be subject to change in future releases." + System.lineSeparator());
        }
        if (this.datastore.equals("redis")) {
            errStream.printf(BalSyntaxConstants.EXPERIMENTAL_NOTICE, "The support for Redis data store is currently an experimental feature, and its behavior may be subject to change in future releases." + System.lineSeparator());
        }
        try {
            schemaFilePath = BalProjectUtils.getSchemaFilePath(this.sourcePath);
        }
        catch (BalException e) {
            errStream.println(e.getMessage());
            return;
        }
        try {
            BalProjectUtils.updateToml(this.sourcePath, this.datastore, moduleNameWithPackage);
            String syntaxTree = this.updateBallerinaToml(Paths.get(this.sourcePath, "Ballerina.toml"), this.datastore, this.testDatastore);
            Utils.writeOutputString(syntaxTree, Paths.get(this.sourcePath, "Ballerina.toml").toAbsolutePath().toString());
            BalProjectUtils.validateSchemaFile(schemaFilePath);
            Object module = BalProjectUtils.getEntities(schemaFilePath);
            if (((Module)module).getEntityMap().isEmpty()) {
                errStream.printf("ERROR: the model definition file(%s) does not contain any entity definition.%n", schemaFilePath.getFileName());
                return;
            }
            entityModule = module;
        }
        catch (BalException | IOException e) {
            errStream.printf("ERROR: Failed to generate types and client for the definition file(%s). %s%n", schemaFilePath.getFileName(), e.getMessage());
            return;
        }
        if (!Files.exists(generatedSourceDirPath, new LinkOption[0])) {
            try {
                Files.createDirectories(generatedSourceDirPath.toAbsolutePath(), new FileAttribute[0]);
            }
            catch (IOException e) {
                errStream.println("ERROR: failed to create the generated directory. " + e.getMessage());
                return;
            }
        }
        SourceGenerator sourceCreator = new SourceGenerator(this.sourcePath, generatedSourceDirPath, moduleNameWithPackage, entityModule);
        try {
            switch (this.datastore) {
                case "mysql": 
                case "mssql": 
                case "postgresql": 
                case "h2": {
                    sourceCreator.createDbSources(this.datastore);
                    break;
                }
                case "googlesheets": {
                    sourceCreator.createGSheetSources();
                    break;
                }
                case "redis": {
                    sourceCreator.createRedisSources();
                    break;
                }
                default: {
                    sourceCreator.createInMemorySources();
                    break;
                }
            }
        }
        catch (BalException e) {
            errStream.printf(String.format(BalSyntaxConstants.ERROR_MSG, this.datastore, e.getMessage()), new Object[0]);
            return;
        }
        errStream.println("Persist client and entity types generated successfully in the " + this.module + " directory.");
        if (this.testDatastore != null) {
            try {
                sourceCreator.createTestDataSources(this.testDatastore);
            }
            catch (BalException e) {
                errStream.printf("ERROR: the test data source creation failed. %s%n", e.getMessage());
                return;
            }
            errStream.printf("The test client for the %s datastore is successfully generated in the %s module.%n", this.testDatastore, this.module);
            BalProjectUtils.printTestClientUsageSteps(this.testDatastore, packageName, this.module);
        }
    }

    public void setParentCmdParser(CommandLine parentCmdParser) {
    }

    private String updateBallerinaToml(Path tomlPath, String datastore, String testDatastore) throws BalException, IOException {
        TomlSyntaxUtils.NativeDependency dependency = TomlSyntaxUtils.getDependencyConfig(datastore, testDatastore);
        TomlSyntaxUtils.ConfigDeclaration declaration = TomlSyntaxUtils.getConfigDeclaration(tomlPath, dependency);
        if (declaration.persistConfigExists()) {
            throw new BalException("persist configuration already exists in the Ballerina.toml. remove the existing configuration and try again.");
        }
        return TomlSyntaxUtils.populateNativeDependencyConfig(datastore, testDatastore, declaration, dependency);
    }

    public String getName() {
        return "persist";
    }

    public void printLongDesc(StringBuilder out) {
        out.append("Generate Client objects corresponding to persist entities").append(System.lineSeparator());
        out.append(System.lineSeparator());
    }

    public void printUsage(StringBuilder stringBuilder) {
        stringBuilder.append("  ballerina persist generate").append(System.lineSeparator());
    }

    public static boolean hasPersistConfig(Path configPath) throws BalException {
        TextDocument configDocument = null;
        try {
            configDocument = TextDocuments.from((String)Files.readString(configPath));
        }
        catch (IOException e) {
            errStream.println("ERROR: failed to read the toml file: " + e.getMessage());
        }
        SyntaxTree syntaxTree = SyntaxTree.from((TextDocument)configDocument);
        DocumentNode rootNote = (DocumentNode)syntaxTree.rootNode();
        NodeList nodeList = rootNote.members();
        boolean dbConfigExists = false;
        for (DocumentMemberDeclarationNode member : nodeList) {
            TableNode node;
            String tableName;
            if (!(member instanceof TableNode) || !(tableName = (node = (TableNode)member).identifier().toSourceCode().trim()).equals("persist")) continue;
            dbConfigExists = true;
            break;
        }
        return dbConfigExists;
    }
}

