/*
 * Decompiled with CFR 0.152.
 */
package io.ballerina.persist.cmd;

import io.ballerina.persist.BalException;
import io.ballerina.persist.PersistToolsConstants;
import io.ballerina.persist.cmd.Utils;
import io.ballerina.persist.models.Module;
import io.ballerina.persist.nodegenerator.SourceGenerator;
import io.ballerina.persist.nodegenerator.syntax.constants.BalSyntaxConstants;
import io.ballerina.persist.nodegenerator.syntax.utils.TomlSyntaxUtils;
import io.ballerina.persist.utils.BalProjectUtils;
import io.ballerina.projects.BallerinaToml;
import io.ballerina.projects.buildtools.CodeGeneratorTool;
import io.ballerina.projects.buildtools.ToolConfig;
import io.ballerina.projects.buildtools.ToolContext;
import io.ballerina.projects.util.ProjectUtils;
import io.ballerina.toml.semantic.diagnostics.TomlNodeLocation;
import io.ballerina.tools.diagnostics.DiagnosticFactory;
import io.ballerina.tools.diagnostics.DiagnosticInfo;
import io.ballerina.tools.diagnostics.Location;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;

@ToolConfig(name="persist")
public class PersistCodeGeneratorTool
implements CodeGeneratorTool {
    private static final PrintStream errStream = System.err;

    public void execute(ToolContext toolContext) {
        TomlNodeLocation location = ((BallerinaToml)toolContext.currentPackage().ballerinaToml().get()).tomlAstNode().location();
        Path projectPath = toolContext.currentPackage().project().sourceRoot();
        Path generatedSourceDirPath = Paths.get(projectPath.toString(), "generated");
        try {
            BalProjectUtils.validateBallerinaProject(projectPath);
            String packageName = TomlSyntaxUtils.readPackageName(projectPath.toString());
            Path schemaFilePath = BalProjectUtils.getSchemaFilePath(projectPath.toString());
            HashMap<String, String> ballerinaTomlConfig = TomlSyntaxUtils.readBallerinaTomlConfig(Paths.get(projectPath.toString(), "Ballerina.toml"));
            String targetModule = ballerinaTomlConfig.get("targetModule").trim();
            String datastore = ballerinaTomlConfig.get("options.datastore").trim();
            String testDatastore = ballerinaTomlConfig.get("options.testDatastore") == null ? null : ballerinaTomlConfig.get("options.testDatastore").trim();
            BalProjectUtils.validateDatastore(datastore);
            BalProjectUtils.validateTestDatastore(datastore, testDatastore);
            if (!targetModule.equals(packageName)) {
                if (!targetModule.startsWith(packageName + ".")) {
                    PersistCodeGeneratorTool.createDiagnostics(toolContext, PersistToolsConstants.DiagnosticMessages.INVALID_MODULE_NAME, (Location)location, ballerinaTomlConfig.get("targetModule"));
                    return;
                }
                String moduleName = targetModule.replace(packageName + ".", "");
                this.validateModuleName(moduleName);
                generatedSourceDirPath = generatedSourceDirPath.resolve(moduleName);
            }
            this.validatePersistDirectory(datastore, projectPath);
            this.printExperimentalFeatureInfo(datastore);
            try {
                if (PersistCodeGeneratorTool.validateCache(toolContext, schemaFilePath) && Files.exists(generatedSourceDirPath, new LinkOption[0])) {
                    return;
                }
            }
            catch (NoSuchAlgorithmException e) {
                errStream.println("INFO: unable to validate the cache. Generating sources for the schema file.");
            }
            BalProjectUtils.validateSchemaFile(schemaFilePath);
            Module entityModule = BalProjectUtils.getEntities(schemaFilePath);
            this.validateEntityModule(entityModule, schemaFilePath);
            String syntaxTree = this.updateBallerinaToml(Paths.get(projectPath.toString(), "Ballerina.toml"), datastore, testDatastore);
            Utils.writeOutputString(syntaxTree, Paths.get(projectPath.toString(), "Ballerina.toml").toAbsolutePath().toString());
            this.createGeneratedSourceDirIfNotExists(generatedSourceDirPath);
            this.generateSources(datastore, entityModule, targetModule, projectPath, generatedSourceDirPath);
            this.generateTestSources(testDatastore, entityModule, targetModule, projectPath, generatedSourceDirPath);
            String modelHashVal = PersistCodeGeneratorTool.getHashValue(schemaFilePath);
            Path cachePath = toolContext.cachePath();
            PersistCodeGeneratorTool.updateCacheFile(cachePath, modelHashVal);
            errStream.println("Persist client and entity types generated successfully in the " + targetModule + " directory.");
        }
        catch (BalException | IOException | NoSuchAlgorithmException e) {
            PersistCodeGeneratorTool.createDiagnostics(toolContext, PersistToolsConstants.DiagnosticMessages.ERROR_WHILE_GENERATING_CLIENT, (Location)location, e.getMessage());
        }
    }

    private String updateBallerinaToml(Path tomlPath, String datastore, String testDatastore) throws BalException, IOException {
        TomlSyntaxUtils.NativeDependency dependency = TomlSyntaxUtils.getDependencyConfig(datastore, testDatastore);
        TomlSyntaxUtils.ConfigDeclaration declaration = TomlSyntaxUtils.getConfigDeclaration(tomlPath, dependency);
        return TomlSyntaxUtils.populateNativeDependencyConfig(datastore, testDatastore, declaration, dependency);
    }

    private void validateModuleName(String moduleName) throws BalException {
        if (!ProjectUtils.validateModuleName((String)moduleName)) {
            throw new BalException("invalid module name : '" + moduleName + "' :" + System.lineSeparator() + "module name can only contain alphanumerics, underscores and periods");
        }
        if (!ProjectUtils.validateNameLength((String)moduleName)) {
            throw new BalException("invalid module name : '" + moduleName + "' :" + System.lineSeparator() + "maximum length of module name is 256 characters");
        }
    }

    private static boolean validateCache(ToolContext toolContext, Path schemaFilePath) throws IOException, NoSuchAlgorithmException {
        Path cachePath = toolContext.cachePath();
        String modelHashVal = PersistCodeGeneratorTool.getHashValue(schemaFilePath);
        if (!Files.isDirectory(cachePath, new LinkOption[0])) {
            return false;
        }
        Path cacheFilePath = Paths.get(cachePath.toString(), "persist-cache.txt");
        String cacheContent = Files.readString(Paths.get(cacheFilePath.toString(), new String[0]));
        return cacheContent.equals(modelHashVal);
    }

    private static void updateCacheFile(Path cachePath, String modelHashVal) {
        try {
            Path cacheFilePath = Paths.get(cachePath.toString(), "persist-cache.txt");
            if (!Files.exists(cacheFilePath, new LinkOption[0])) {
                Files.createDirectories(cachePath, new FileAttribute[0]);
                Files.createFile(cacheFilePath, new FileAttribute[0]);
            }
            Files.writeString(cacheFilePath, (CharSequence)modelHashVal, StandardCharsets.UTF_8, StandardOpenOption.WRITE);
        }
        catch (IOException e) {
            errStream.println("ERROR: failed to update the cache file: " + e.getMessage());
        }
    }

    private static String getHashValue(Path schemaFilePath) throws IOException, NoSuchAlgorithmException {
        String schema = PersistCodeGeneratorTool.readFileToString(schemaFilePath);
        MessageDigest messageDigest = MessageDigest.getInstance("SHA-256");
        byte[] hashBytes = messageDigest.digest(schema.getBytes(StandardCharsets.UTF_8));
        return new String(hashBytes, StandardCharsets.UTF_8);
    }

    public static String readFileToString(Path filePath) throws IOException {
        byte[] fileContent = Files.readAllBytes(filePath);
        return new String(fileContent, StandardCharsets.UTF_8);
    }

    private void validatePersistDirectory(String datastore, Path projectPath) throws BalException {
        if (Files.isDirectory(Paths.get(projectPath.toString(), "persist", "migrations"), new LinkOption[0]) && !datastore.equals("mysql")) {
            throw new BalException("regenerating the client with a different datastore after executing the migrate command is not permitted. please remove the migrations directory within the persist directory and try executing the command again.");
        }
    }

    private void printExperimentalFeatureInfo(String datastore) {
        if (datastore.equals("googlesheets")) {
            errStream.printf(BalSyntaxConstants.EXPERIMENTAL_NOTICE, "The support for Google Sheets data store is currently an experimental feature, and its behavior may be subject to change in future releases." + System.lineSeparator());
        }
    }

    private void validateEntityModule(Module entityModule, Path schemaFilePath) throws BalException {
        if (entityModule.getEntityMap().isEmpty()) {
            throw new BalException(String.format("the model definition file(%s) does not contain any entity definition.", schemaFilePath.getFileName()));
        }
    }

    private void createGeneratedSourceDirIfNotExists(Path generatedSourceDirPath) throws IOException {
        if (!Files.exists(generatedSourceDirPath, new LinkOption[0])) {
            Files.createDirectories(generatedSourceDirPath.toAbsolutePath(), new FileAttribute[0]);
        }
    }

    private void generateSources(String datastore, Module entityModule, String targetModule, Path projectPath, Path generatedSourceDirPath) throws BalException {
        SourceGenerator sourceCreator = new SourceGenerator(projectPath.toString(), generatedSourceDirPath, targetModule, entityModule);
        switch (datastore) {
            case "mysql": 
            case "mssql": 
            case "postgresql": 
            case "h2": {
                sourceCreator.createDbSources(datastore);
                break;
            }
            case "googlesheets": {
                sourceCreator.createGSheetSources();
                break;
            }
            case "redis": {
                sourceCreator.createRedisSources();
                break;
            }
            default: {
                sourceCreator.createInMemorySources();
            }
        }
    }

    private void generateTestSources(String testDatastore, Module entityModule, String targetModule, Path projectPath, Path generatedSourceDirPath) {
        if (testDatastore != null) {
            try {
                SourceGenerator sourceCreator = new SourceGenerator(projectPath.toString(), generatedSourceDirPath, targetModule, entityModule);
                sourceCreator.createTestDataSources(testDatastore);
            }
            catch (BalException e) {
                errStream.printf("ERROR: the test data source creation failed. %s%n", e.getMessage());
            }
        }
    }

    private static void createDiagnostics(ToolContext toolContext, PersistToolsConstants.DiagnosticMessages error, Location location, String ... args) {
        String message = String.format(error.getDescription(), args);
        DiagnosticInfo diagnosticInfo = new DiagnosticInfo(error.getCode(), message, error.getSeverity());
        toolContext.reportDiagnostic(DiagnosticFactory.createDiagnostic((DiagnosticInfo)diagnosticInfo, (Location)location, (Object[])new Object[0]));
    }
}

