/*
 * Decompiled with CFR 0.152.
 */
package org.wso2.transport.localfilesystem.server;

import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wso2.transport.localfilesystem.server.connector.contract.LocalFileSystemEvent;
import org.wso2.transport.localfilesystem.server.connector.contract.LocalFileSystemListener;
import org.wso2.transport.localfilesystem.server.exception.LocalFileSystemServerConnectorException;

public class DirectoryListener
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(DirectoryListener.class);
    private final WatchService watcher;
    private final Map<WatchKey, Path> keys;
    private final boolean recursive;
    private final String serviceName;
    private ExecutorService executorService;
    private final WatchEvent.Kind[] registeredEvents;
    private LocalFileSystemListener localFileSystemListener;

    public DirectoryListener(String id, Map<String, String> config, LocalFileSystemListener listener) throws LocalFileSystemServerConnectorException {
        this.serviceName = id;
        this.localFileSystemListener = listener;
        String path = config.get("dirURI");
        if (path == null || path.isEmpty()) {
            throw new LocalFileSystemServerConnectorException("Directory path[dirURI] property empty or not available for service: " + this.serviceName);
        }
        String eventProperty = config.get("events");
        if (eventProperty == null || eventProperty.isEmpty()) {
            throw new LocalFileSystemServerConnectorException("Listener events are not specified in 'events' property");
        }
        this.registeredEvents = this.getEventArray(eventProperty);
        try {
            this.watcher = FileSystems.getDefault().newWatchService();
            this.keys = new HashMap<WatchKey, Path>();
            this.recursive = Boolean.parseBoolean(config.get("recursive"));
            Path dir = Paths.get(path, new String[0]);
            if (this.recursive) {
                this.registerAll(dir);
            } else {
                this.register(dir);
            }
        }
        catch (IOException e) {
            throw new LocalFileSystemServerConnectorException("Unable to get a watch directory for service: " + this.serviceName, e);
        }
    }

    public void start() {
        this.executorService = Executors.newSingleThreadExecutor();
        this.executorService.execute(this);
        if (log.isDebugEnabled()) {
            log.debug("Successfully start directory listen for service: " + this.serviceName);
        }
    }

    public void stop() throws LocalFileSystemServerConnectorException {
        try {
            this.watcher.close();
        }
        catch (IOException e) {
            throw new LocalFileSystemServerConnectorException("Unable to stop watching for service: " + this.serviceName, e);
        }
        if (this.executorService != null) {
            this.executorService.shutdown();
            try {
                if (!this.executorService.awaitTermination(500L, TimeUnit.MILLISECONDS)) {
                    this.executorService.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.executorService.shutdownNow();
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("Successfully stop directory listen for service: " + this.serviceName);
        }
    }

    @Override
    public void run() {
        this.startWatch();
    }

    private void registerAll(Path start) throws IOException {
        Files.walkFileTree(start, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attr) throws IOException {
                DirectoryListener.this.register(dir);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private void register(Path dir) throws IOException {
        WatchKey key = dir.register(this.watcher, this.registeredEvents);
        this.keys.put(key, dir);
    }

    private WatchEvent.Kind[] getEventArray(String eventProperty) throws LocalFileSystemServerConnectorException {
        String[] eventArray;
        ArrayList<WatchEvent.Kind<Path>> events = new ArrayList<WatchEvent.Kind<Path>>(3);
        block10: for (String event : eventArray = eventProperty.split("\\s*,\\s*")) {
            switch (event.toLowerCase(Locale.getDefault())) {
                case "create": {
                    events.add(StandardWatchEventKinds.ENTRY_CREATE);
                    continue block10;
                }
                case "delete": {
                    events.add(StandardWatchEventKinds.ENTRY_DELETE);
                    continue block10;
                }
                case "modify": {
                    events.add(StandardWatchEventKinds.ENTRY_MODIFY);
                    continue block10;
                }
                default: {
                    throw new LocalFileSystemServerConnectorException("Unidentified event type try to register: " + event);
                }
            }
        }
        return events.toArray(new WatchEvent.Kind[events.size()]);
    }

    private void startWatch() {
        while (true) {
            WatchKey key;
            try {
                key = this.watcher.take();
            }
            catch (InterruptedException x) {
                return;
            }
            Path dir = this.keys.get(key);
            if (dir == null) {
                if (!log.isDebugEnabled()) continue;
                log.debug("Received [" + key + "] for unknown directory.");
                continue;
            }
            this.processEvent(key, dir);
            if (key.reset()) continue;
            this.keys.remove(key);
        }
    }

    private void processEvent(WatchKey key, Path dir) {
        for (WatchEvent<?> event : key.pollEvents()) {
            WatchEvent.Kind<?> kind = event.kind();
            if (kind == StandardWatchEventKinds.OVERFLOW) {
                if (!log.isDebugEnabled()) continue;
                log.debug("OVERFLOW event received for service: " + this.serviceName);
                continue;
            }
            WatchEvent ev = DirectoryListener.cast(event);
            Path child = dir.resolve((Path)ev.context());
            this.notifyToListener(event, child);
            if (!this.recursive || kind != StandardWatchEventKinds.ENTRY_CREATE) continue;
            try {
                if (!Files.isDirectory(child, LinkOption.NOFOLLOW_LINKS)) continue;
                this.registerAll(child);
            }
            catch (IOException e) {
                log.warn("Unable listen to the newly created directory: " + child + " in service " + this.serviceName, (Throwable)e);
            }
        }
    }

    private void notifyToListener(WatchEvent<?> event, Path child) {
        String eventType = null;
        switch (event.kind().name()) {
            case "ENTRY_CREATE": {
                eventType = "create";
                break;
            }
            case "ENTRY_DELETE": {
                eventType = "delete";
                break;
            }
            case "ENTRY_MODIFY": {
                eventType = "modify";
                break;
            }
        }
        LocalFileSystemEvent message = new LocalFileSystemEvent(child.toString(), eventType);
        message.setProperty("TRANSPORT_FILE_SERVICE_NAME", this.serviceName);
        this.localFileSystemListener.onMessage(message);
    }

    private static <T> WatchEvent<T> cast(WatchEvent<?> event) {
        return event;
    }
}

