/*
 * Decompiled with CFR 0.152.
 */
package io.ballerina.cli.utils;

import io.ballerina.cli.cmd.RunCommand;
import io.ballerina.cli.utils.RunCommandExecutor;
import io.ballerina.projects.ProjectKind;
import io.ballerina.projects.internal.ProjectFiles;
import io.ballerina.projects.util.FileUtils;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public final class ProjectWatcher {
    private final WatchService fileWatcher;
    private final Map<WatchKey, Path> watchKeys;
    private final RunCommand runCommand;
    private final PrintStream outStream;
    private final Path projectPath;
    private final ProjectKind projectKind;
    private final ScheduledExecutorService scheduledExecutorService;
    private final Map<Path, Long> debounceMap = new ConcurrentHashMap<Path, Long>();
    private static final long debounceTimeMillis = 250L;
    private final RunCommandExecutor[] thread;
    private volatile boolean forceStop = false;

    public ProjectWatcher(RunCommand runCommand, Path projectPath, PrintStream outStream) throws IOException {
        this.fileWatcher = FileSystems.getDefault().newWatchService();
        this.runCommand = runCommand;
        this.thread = new RunCommandExecutor[]{new RunCommandExecutor(runCommand, outStream)};
        this.projectPath = projectPath.toAbsolutePath();
        this.outStream = outStream;
        this.watchKeys = new HashMap<WatchKey, Path>();
        this.projectKind = this.deriveProjectKind();
        this.validateProjectPath();
        this.scheduledExecutorService = Executors.newScheduledThreadPool(1);
        this.registerFileTree(projectPath);
    }

    public void watch() throws IOException {
        this.thread[0].start();
        while (this.thread[0].shouldWatch() && !this.forceStop) {
            WatchKey key = this.fileWatcher.poll();
            Path dir = this.watchKeys.get(key);
            if (dir == null) continue;
            for (WatchEvent<?> event : key.pollEvents()) {
                WatchEvent.Kind<?> kind = event.kind();
                if (kind == StandardWatchEventKinds.OVERFLOW) continue;
                WatchEvent pathWatchEvent = ProjectWatcher.cast(event);
                Path changedFileName = (Path)pathWatchEvent.context();
                Path changedFilePath = dir.resolve(changedFileName).toAbsolutePath();
                if (this.isValidFileChange(changedFilePath)) {
                    long currentTime = System.currentTimeMillis();
                    this.debounceMap.put(changedFilePath, currentTime);
                    this.scheduledExecutorService.schedule(() -> {
                        Long lastModifiedTime = this.debounceMap.get(changedFilePath);
                        if (lastModifiedTime == null || System.currentTimeMillis() - lastModifiedTime < 250L) {
                            return;
                        }
                        this.outStream.println("\nDetected file changes. Re-running the project...");
                        this.thread[0].terminate();
                        this.waitForRunCmdThreadToJoin();
                        this.thread[0] = new RunCommandExecutor(this.runCommand, this.outStream);
                        this.thread[0].start();
                        this.debounceMap.remove(changedFilePath);
                    }, 250L, TimeUnit.MILLISECONDS);
                }
                if (kind != StandardWatchEventKinds.ENTRY_CREATE || !Files.isDirectory(changedFilePath, new LinkOption[0])) continue;
                this.registerFileTree(changedFilePath);
            }
            boolean valid = key.reset();
            if (valid) continue;
            this.watchKeys.remove(key);
            if (!this.watchKeys.isEmpty()) continue;
            break;
        }
        this.waitForRunCmdThreadToJoin();
    }

    public void stopWatching() {
        try {
            if (this.thread != null) {
                this.thread[0].terminate();
                this.thread[0].join();
            }
            this.forceStop = true;
            this.fileWatcher.close();
        }
        catch (IOException | InterruptedException e) {
            this.outStream.println("Error occurred while stopping the project watcher: " + e.getMessage());
        }
    }

    private ProjectKind deriveProjectKind() {
        return FileUtils.hasExtension((Path)this.projectPath) ? ProjectKind.SINGLE_FILE_PROJECT : ProjectKind.BUILD_PROJECT;
    }

    private boolean isValidFileChange(Path path) {
        if (this.projectKind.equals((Object)ProjectKind.SINGLE_FILE_PROJECT)) {
            return this.projectPath.equals(path);
        }
        if (Files.isDirectory(path = this.projectPath.relativize(path), new LinkOption[0])) {
            return false;
        }
        Path fileNamePath = path.getFileName();
        if (fileNamePath == null) {
            return false;
        }
        String fileName = fileNamePath.toString();
        if (path.getNameCount() == 1) {
            if (fileName.endsWith(".bal")) {
                return true;
            }
            return fileName.endsWith(".toml") && !fileName.equals("Dependencies.toml");
        }
        if (path.startsWith("resources") && path.getNameCount() > 1) {
            return true;
        }
        if (path.startsWith("modules") && path.getNameCount() > 2) {
            Path modulePath = path.subpath(2, path.getNameCount());
            if (modulePath.getNameCount() == 1 && fileName.endsWith(".bal")) {
                return true;
            }
            return modulePath.startsWith("resources");
        }
        return false;
    }

    private void registerFileTree(Path path) throws IOException {
        if (this.projectKind.equals((Object)ProjectKind.SINGLE_FILE_PROJECT)) {
            Path parentPath = path.toAbsolutePath().getParent();
            if (parentPath != null) {
                this.register(parentPath);
            }
            return;
        }
        Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                ProjectWatcher.this.register(dir);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private void register(Path dir) throws IOException {
        WatchKey key = dir.register(this.fileWatcher, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
        this.watchKeys.put(key, dir);
    }

    private void validateProjectPath() {
        if (this.projectKind.equals((Object)ProjectKind.SINGLE_FILE_PROJECT)) {
            ProjectFiles.validateSingleFileProjectFilePath((Path)this.projectPath);
        } else {
            ProjectFiles.validateBuildProjectDirPath((Path)this.projectPath);
        }
    }

    private static <T> WatchEvent<T> cast(WatchEvent<?> event) {
        return event;
    }

    private void waitForRunCmdThreadToJoin() {
        try {
            this.thread[0].join();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }
}

