/*
 * Decompiled with CFR 0.152.
 */
package io.ballerina.persist.cmd;

import io.ballerina.cli.BLauncherCmd;
import io.ballerina.persist.BalException;
import io.ballerina.persist.models.Entity;
import io.ballerina.persist.models.EntityField;
import io.ballerina.persist.models.ForeignKey;
import io.ballerina.persist.models.Index;
import io.ballerina.persist.models.MigrationDataHolder;
import io.ballerina.persist.models.Module;
import io.ballerina.persist.models.Relation;
import io.ballerina.persist.nodegenerator.SourceGenerator;
import io.ballerina.persist.nodegenerator.syntax.constants.BalSyntaxConstants;
import io.ballerina.persist.nodegenerator.syntax.utils.SqlScriptUtils;
import io.ballerina.persist.utils.BalProjectUtils;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;
import picocli.CommandLine;

@CommandLine.Command(name="migrate", description={"Generate DB migration scripts for Ballerina schema changes."})
public class Migrate
implements BLauncherCmd {
    private static final PrintStream errStream = System.err;
    private final String sourcePath;
    private static final String COMMAND_IDENTIFIER = "persist-migrate";
    @CommandLine.Parameters
    public List<String> argList;
    @CommandLine.Option(names={"--datastore"})
    private String datastore = "mysql";
    @CommandLine.Option(names={"-h", "--help"}, hidden=true)
    private boolean helpFlag;

    public Migrate() {
        this("");
    }

    public Migrate(String sourcePath) {
        this.sourcePath = sourcePath;
    }

    public void execute() {
        Path schemaFilePath;
        if (this.helpFlag) {
            String commandUsageInfo = BLauncherCmd.getCommandUsageInfo((String)COMMAND_IDENTIFIER, (ClassLoader)Migrate.class.getClassLoader());
            errStream.println(commandUsageInfo);
            return;
        }
        errStream.printf(BalSyntaxConstants.EXPERIMENTAL_NOTICE, "The support for migrations is currently an experimental feature, and its behavior may be subject to change in future releases.");
        Path projectPath = Paths.get(this.sourcePath, new String[0]).toAbsolutePath();
        try {
            BalProjectUtils.validateBallerinaProject(projectPath);
        }
        catch (BalException e) {
            errStream.println(e.getMessage());
            return;
        }
        if (!Objects.equals(this.datastore, "mysql")) {
            errStream.println("Error: invalid datastore: " + this.datastore + ". currently only MySQL is supported.");
            return;
        }
        if (null == this.argList || this.argList.isEmpty()) {
            errStream.println("Error: migration label is not provided. Provide the migration label along with the command like `bal persist migrate <migration-label>`");
            return;
        }
        if (1 < this.argList.size()) {
            errStream.println("Error: Too many arguments provided. Only one argument is allowed to pass for migration label");
            return;
        }
        String migrationName = this.argList.get(0);
        try {
            schemaFilePath = BalProjectUtils.getSchemaFilePath(this.sourcePath);
        }
        catch (BalException e) {
            errStream.println(e.getMessage());
            return;
        }
        Migrate.migrate(migrationName, projectPath, this.sourcePath, schemaFilePath);
    }

    private static void migrate(String migrationName, Path projectDirPath, String sourcePath, Path schemaFilePath) {
        if (schemaFilePath != null) {
            Path persistDirPath = Paths.get(projectDirPath.toString(), "persist");
            File persistDir = new File(persistDirPath.toString());
            File migrationsDir = new File(persistDir, "migrations");
            if (!migrationsDir.exists()) {
                Module model;
                boolean created = migrationsDir.mkdir();
                if (!created) {
                    errStream.println("Error: failed to create migrations directory inside the persist directory");
                    return;
                }
                try {
                    model = BalProjectUtils.getEntities(schemaFilePath);
                }
                catch (BalException e) {
                    errStream.println("Error getting entities: " + e.getMessage());
                    return;
                }
                String newMigration = Migrate.createTimestampFolder(migrationName, migrationsDir);
                Migrate.createTimestampDirectory(newMigration);
                Path newMigrationPath = Paths.get(newMigration, new String[0]);
                if (!model.getEntityMap().isEmpty()) {
                    try {
                        SourceGenerator.addSqlScriptFile("the migrate command", SqlScriptUtils.generateSqlScript(model.getEntityMap().values(), "mysql"), newMigrationPath);
                    }
                    catch (BalException e) {
                        errStream.println("ERROR: failed to generate SQL script " + e.getMessage());
                        return;
                    }
                    try {
                        Files.copy(schemaFilePath, newMigrationPath.resolve(schemaFilePath.getFileName()), new CopyOption[0]);
                    }
                    catch (IOException e) {
                        errStream.println("Error: Copying file failed: " + e.getMessage());
                        return;
                    }
                    Path relativePath = Paths.get("", new String[0]).toAbsolutePath().relativize(newMigrationPath);
                    ArrayList<String> differences = new ArrayList<String>();
                    differences.add("Table " + String.join((CharSequence)", ", model.getEntityMap().keySet()) + " has been added");
                    Migrate.printDetailedListOfDifferences(differences);
                    errStream.println("Generated migration script to " + String.valueOf(relativePath) + " directory." + System.lineSeparator());
                    errStream.println("Next steps:" + System.lineSeparator() + "Execute the \"script.sql\" file located at " + String.valueOf(relativePath) + " directory in your database to migrate the schema with the latest changes.");
                } else {
                    errStream.println("ERROR: Could not find any entities in the schema file");
                }
            } else {
                Migrate.migrateWithTimestamp(migrationsDir, migrationName, schemaFilePath, Migrate.findLatestBalFile(Migrate.getDirectoryPaths(migrationsDir.toString()), sourcePath));
            }
        }
    }

    private static List<String> getDirectoryPaths(String folder) {
        ArrayList<String> directoryNames = new ArrayList<String>();
        Path folderPath = Path.of(folder, new String[0]);
        if (Files.exists(folderPath, new LinkOption[0]) && Files.isDirectory(folderPath, new LinkOption[0])) {
            try (Stream<Path> directoryStream = Files.list(folderPath);){
                List<Path> directories = directoryStream.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).toList();
                for (Path directory : directories) {
                    Path fileName = directory.getFileName();
                    if (fileName != null) {
                        directoryNames.add(fileName.toString());
                        continue;
                    }
                    errStream.println("Error: Found a directory with a null file name.");
                }
            }
            catch (IOException e) {
                errStream.println("Error: An error occurred while retrieving the directory paths: " + e.getMessage());
            }
        }
        return directoryNames;
    }

    private static Path findLatestBalFile(List<String> folderNames, String sourcePath) {
        if (folderNames.size() == 1) {
            return Migrate.findBalFileInFolder(folderNames.get(0), sourcePath);
        }
        String latestTimestamp = "";
        ZonedDateTime latestDateTime = ZonedDateTime.ofInstant(Instant.EPOCH, ZoneOffset.UTC);
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMddHHmmss").withZone(ZoneOffset.UTC);
        for (String folderName : folderNames) {
            String timestamp = folderName.split("_")[0];
            ZonedDateTime dateTime = ZonedDateTime.parse(timestamp, formatter);
            if (!dateTime.isAfter(latestDateTime)) continue;
            latestDateTime = dateTime;
            latestTimestamp = folderName;
        }
        return Migrate.findBalFileInFolder(latestTimestamp, sourcePath);
    }

    private static Path findBalFileInFolder(String folderName, String sourcePath) {
        File balFile;
        Path folderPath = Paths.get(sourcePath, new String[0]).resolve("persist").resolve("migrations").resolve(folderName).toAbsolutePath();
        File folder = folderPath.toFile();
        File[] files = folder.listFiles();
        if (files != null && (balFile = (File)Arrays.stream(files).filter(file -> file.isFile() && file.getName().endsWith(".bal")).findFirst().orElse(null)) != null) {
            return balFile.toPath().toAbsolutePath();
        }
        return null;
    }

    private static void migrateWithTimestamp(File migrationsDir, String migrationName, Path currentModelPath, Path previousModelPath) {
        List<String> queries;
        Path newMigrationPath;
        File newMigrateDirectory;
        block28: {
            String newMigration = Migrate.createTimestampFolder(migrationName, migrationsDir);
            newMigrateDirectory = Migrate.getTimestampDirectory(newMigration);
            newMigrationPath = Paths.get(newMigration, new String[0]);
            try {
                Module previousModel = BalProjectUtils.getEntities(previousModelPath);
                Module currentModel = BalProjectUtils.getEntities(currentModelPath);
                queries = Migrate.findDifferences(previousModel, currentModel);
                if (queries.isEmpty()) break block28;
                String filePath = Paths.get(newMigrationPath.toString(), "script.sql").toString();
                try (FileOutputStream fStream = new FileOutputStream(filePath);
                     OutputStreamWriter oStream = new OutputStreamWriter((OutputStream)fStream, StandardCharsets.UTF_8);
                     BufferedWriter writer = new BufferedWriter(oStream);){
                    writer.write("-- AUTO-GENERATED FILE." + System.lineSeparator() + "-- This file is an auto-generated file by Ballerina persistence layer for the migrate command." + System.lineSeparator() + "-- Please verify the generated scripts and execute them against the target DB server." + System.lineSeparator() + System.lineSeparator());
                    for (String query : queries) {
                        writer.write(query);
                        writer.newLine();
                    }
                }
                catch (IOException e) {
                    errStream.println("Error: An error occurred while writing to file: " + e.getMessage());
                    return;
                }
                Path relativePath = Paths.get("", new String[0]).toAbsolutePath().relativize(newMigrationPath);
                errStream.println("Generated migration script to " + String.valueOf(relativePath) + " directory." + System.lineSeparator());
                errStream.println("Next steps:" + System.lineSeparator() + "Execute the \"script.sql\" file located at " + String.valueOf(relativePath) + " directory in your database to migrate the schema with the latest changes.");
            }
            catch (BalException e) {
                errStream.println("Error getting entities: " + e.getMessage());
                return;
            }
        }
        if (!queries.isEmpty()) {
            try {
                Files.copy(currentModelPath, newMigrationPath.resolve(currentModelPath.getFileName()), new CopyOption[0]);
            }
            catch (IOException e) {
                errStream.println("Error: Copying file failed: " + e.getMessage());
            }
        } else {
            File[] files;
            boolean deleteNewMigrateFolder;
            if (newMigrateDirectory != null && !(deleteNewMigrateFolder = newMigrateDirectory.delete())) {
                errStream.println("Error: Failed to delete timestamp folder.");
            }
            if (migrationsDir.exists() && migrationsDir.isDirectory() && (files = migrationsDir.listFiles()) != null && files.length == 0) {
                try {
                    Files.delete(migrationsDir.toPath());
                }
                catch (IOException e) {
                    errStream.println("Error: Failed to delete migration folder: " + e.getMessage());
                }
            }
        }
    }

    private static String createTimestampFolder(String migrationName, File migrationsDir) {
        Instant currentTime = Instant.now();
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");
        String timestamp = formatter.format(ZonedDateTime.ofInstant(currentTime, ZoneOffset.UTC));
        return String.valueOf(migrationsDir) + File.separator + timestamp + "_" + migrationName;
    }

    private static File getTimestampDirectory(String newMigration) {
        File newMigrateDirectory = new File(newMigration);
        if (!newMigrateDirectory.exists()) {
            boolean isDirectoryCreated = newMigrateDirectory.mkdirs();
            if (!isDirectoryCreated) {
                errStream.println("Error: Failed to create directory: " + newMigrateDirectory.getAbsolutePath());
            } else {
                return newMigrateDirectory;
            }
        }
        return null;
    }

    private static void createTimestampDirectory(String newMigration) {
        boolean isDirectoryCreated;
        File newMigrateDirectory = new File(newMigration);
        if (!newMigrateDirectory.exists() && !(isDirectoryCreated = newMigrateDirectory.mkdirs())) {
            errStream.println("Error: Failed to create directory: " + newMigrateDirectory.getAbsolutePath());
        }
    }

    private static List<String> findDifferences(Module previousModel, Module currentModel) {
        ArrayList<String> queries = new ArrayList<String>();
        MigrationDataHolder migrationDataHolder = new MigrationDataHolder();
        for (Entity previousModelEntity : previousModel.getEntityMap().values()) {
            Entity currentModelEntity = currentModel.getEntityMap().get(previousModelEntity.getEntityName());
            if (currentModelEntity == null) {
                migrationDataHolder.removeTable(previousModelEntity.getTableName());
                continue;
            }
            if (!Objects.equals(currentModelEntity.getTableName(), previousModelEntity.getTableName())) {
                migrationDataHolder.renameTable(previousModelEntity.getTableName(), currentModelEntity.getTableName());
            }
            if (previousModelEntity.getKeys().size() > currentModelEntity.getKeys().size()) {
                migrationDataHolder.changePrimaryKey(currentModelEntity.getTableName());
            }
            for (EntityField previousModelField : previousModelEntity.getFields()) {
                EntityField currentModelField = currentModelEntity.getFieldByName(previousModelField.getFieldName());
                if (currentModelField == null) {
                    if (previousModelField.getRelation() == null) {
                        migrationDataHolder.removeColumn(previousModelEntity.getTableName(), previousModelField.getFieldColumnName());
                        continue;
                    }
                    if (!previousModelField.getRelation().isOwner()) continue;
                    migrationDataHolder.removeForeignKey(previousModelEntity.getTableName(), previousModelField);
                    continue;
                }
                if (!Objects.equals(currentModelField.getFieldColumnName(), previousModelField.getFieldColumnName())) {
                    migrationDataHolder.renameColumn(currentModelEntity.getTableName(), previousModelField.getFieldColumnName(), currentModelField.getFieldColumnName());
                }
                if (currentModelField.getRelation() != null && previousModelField.getRelation() != null && currentModelField.getRelation().isOwner() && !Objects.equals(currentModelField.getRelation().getKeyColumns(), previousModelField.getRelation().getKeyColumns())) {
                    if (Migrate.isOnlyColumnsRenamed(previousModelField.getRelation().getKeyColumns(), currentModelField.getRelation().getKeyColumns())) {
                        for (int i = 0; i < previousModelField.getRelation().getKeyColumns().size(); ++i) {
                            migrationDataHolder.renameColumn(currentModelEntity.getTableName(), previousModelField.getRelation().getKeyColumns().get(i).getColumnName(), currentModelField.getRelation().getKeyColumns().get(i).getColumnName());
                        }
                    } else {
                        migrationDataHolder.recreateForeignKey(currentModelEntity.getTableName(), previousModelField, currentModelField);
                    }
                }
                if (!(previousModelField.getFieldType().equals(currentModelField.getFieldType()) && Objects.equals(previousModelField.getSqlType(), currentModelField.getSqlType()) && Objects.equals(previousModelField.isOptionalType(), currentModelField.isOptionalType()) && Objects.equals(previousModelField.isDbGenerated(), currentModelField.isDbGenerated()))) {
                    migrationDataHolder.modifyColumn(currentModelEntity.getTableName(), previousModelField, currentModelField);
                }
                if (previousModelEntity.getKeys().contains(previousModelField) || !currentModelEntity.getKeys().contains(currentModelField)) continue;
                migrationDataHolder.changePrimaryKey(currentModelEntity.getTableName());
            }
            for (EntityField currentModelField : currentModelEntity.getFields()) {
                EntityField previousModelField = previousModelEntity.getFieldByName(currentModelField.getFieldName());
                if (previousModelField != null) continue;
                if (currentModelField.getRelation() == null) {
                    migrationDataHolder.addColumn(currentModelEntity.getTableName(), currentModelField, currentModelEntity.getKeys().contains(currentModelField));
                    continue;
                }
                if (!currentModelField.getRelation().isOwner()) continue;
                migrationDataHolder.createForeignKeys(currentModelEntity.getTableName(), currentModelField);
            }
        }
        for (Entity currentModelEntity : currentModel.getEntityMap().values()) {
            Entity previousModelEntity = previousModel.getEntityMap().get(currentModelEntity.getEntityName());
            if (previousModelEntity != null || migrationDataHolder.isEntityRenamed(currentModelEntity.getTableName())) continue;
            migrationDataHolder.addTable(currentModelEntity.getTableName());
        }
        HashMap<String, List<Index>> previousIndexes = Migrate.getIndexesFromModule(previousModel);
        HashMap<String, List<Index>> currentIndexes = Migrate.getIndexesFromModule(currentModel);
        Migrate.processIndexDifferences(previousIndexes, currentIndexes, migrationDataHolder);
        HashMap<String, List<Index>> previousUniqueIndexes = Migrate.getUniqueIndexesFromModule(previousModel);
        HashMap<String, List<Index>> currentUniqueIndexes = Migrate.getUniqueIndexesFromModule(currentModel);
        Migrate.processIndexDifferences(previousUniqueIndexes, currentUniqueIndexes, migrationDataHolder);
        Migrate.addDropTableQueries(migrationDataHolder.getRemovedEntities(), queries);
        Migrate.addDropForeignKeyQueries(migrationDataHolder.getRemovedForeignKeys(), queries);
        Migrate.addDropPrimaryKeyQueries(migrationDataHolder.getPrimaryKeyChangedEntities(), migrationDataHolder.getAddedEntities(), queries);
        Migrate.addDropColumnQueries(migrationDataHolder.getRemovedFields(), queries);
        Migrate.addCreateTableQueries(migrationDataHolder.getAddedEntities(), currentModel, queries);
        Migrate.addRenameTableQueries(migrationDataHolder.getRenamedEntities(), queries);
        Migrate.addRenameFieldQueries(migrationDataHolder.getRenamedFields(), queries);
        Migrate.addCreateFieldQueries(migrationDataHolder.getAddedFields(), queries);
        Migrate.addCreatePrimaryKeyQueries(migrationDataHolder.getPrimaryKeyChangedEntities(), migrationDataHolder.getAddedEntities(), currentModel, queries);
        Migrate.addCreateForeignKeyQueries(migrationDataHolder.getAddedForeignKeys(), queries);
        Migrate.addModifyColumnTypeQueries(migrationDataHolder.getChangedFieldTypes(), queries);
        Migrate.addDropIndexQueries(migrationDataHolder.getRemovedIndexes(), queries);
        Migrate.addCreateIndexQueries(migrationDataHolder.getAddedIndexes(), queries);
        Migrate.printDetailedListOfDifferences(migrationDataHolder.getDifferences());
        return queries;
    }

    private static void processIndexDifferences(HashMap<String, List<Index>> previousIndexes, HashMap<String, List<Index>> currentIndexes, MigrationDataHolder migrationDataHolder) {
        for (Map.Entry<String, List<Index>> entry : previousIndexes.entrySet()) {
            if (migrationDataHolder.getRemovedEntities().contains(entry.getKey())) continue;
            List<Index> previousIndexList = entry.getValue();
            List<Index> currentIndexList = currentIndexes.get(entry.getKey());
            if (Objects.isNull(currentIndexList)) {
                for (Index index : previousIndexList) {
                    migrationDataHolder.removeIndex(entry.getKey(), index);
                }
                continue;
            }
            for (Index previousIndex2 : previousIndexList) {
                boolean isIndexFound = false;
                block3: for (Index currentIndex : currentIndexList) {
                    if (!previousIndex2.getIndexName().equals(currentIndex.getIndexName())) continue;
                    isIndexFound = true;
                    if (!Objects.equals(currentIndex.getFields().size(), previousIndex2.getFields().size())) {
                        migrationDataHolder.removeIndex(entry.getKey(), previousIndex2);
                        migrationDataHolder.addIndex(entry.getKey(), currentIndex);
                        break;
                    }
                    for (int i = 0; i < currentIndex.getFields().size(); ++i) {
                        if (currentIndex.getFields().get(i).getFieldName().equals(previousIndex2.getFields().get(i).getFieldName())) continue;
                        migrationDataHolder.removeIndex(entry.getKey(), previousIndex2);
                        migrationDataHolder.addIndex(entry.getKey(), currentIndex);
                        break block3;
                    }
                }
                if (isIndexFound) continue;
                migrationDataHolder.removeIndex(entry.getKey(), previousIndex2);
            }
        }
        for (Map.Entry<String, List<Index>> entry : currentIndexes.entrySet()) {
            String entity = entry.getKey();
            for (Index index : entry.getValue()) {
                if (previousIndexes.get(entity) != null && !previousIndexes.get(entity).stream().noneMatch(previousIndex -> previousIndex.getIndexName().equals(index.getIndexName()))) continue;
                migrationDataHolder.addIndex(entity, index);
            }
        }
    }

    private static boolean isOnlyColumnsRenamed(List<Relation.Key> previousKeys, List<Relation.Key> currentKeys) {
        if (!Objects.equals(previousKeys.size(), currentKeys.size())) {
            return false;
        }
        for (int i = 0; i < previousKeys.size(); ++i) {
            if (previousKeys.get(i).isOnlyColumnRenamed(currentKeys.get(i))) continue;
            return false;
        }
        return true;
    }

    private static void printDetailedListOfDifferences(List<String> differences) {
        errStream.println(System.lineSeparator() + "Detailed list of differences: ");
        if (!differences.isEmpty()) {
            differences.forEach(difference -> errStream.println("-- " + difference));
            errStream.println();
        } else {
            errStream.println("-- No differences found" + System.lineSeparator());
        }
    }

    private static HashMap<String, List<Index>> getIndexesFromModule(Module module) {
        HashMap<String, List<Index>> indexMap = new HashMap<String, List<Index>>();
        for (Entity entity : module.getEntityMap().values()) {
            if (entity.getIndexes().isEmpty()) continue;
            indexMap.put(entity.getTableName(), entity.getIndexes());
        }
        return indexMap;
    }

    private static HashMap<String, List<Index>> getUniqueIndexesFromModule(Module module) {
        HashMap<String, List<Index>> indexMap = new HashMap<String, List<Index>>();
        for (Entity entity : module.getEntityMap().values()) {
            if (entity.getUniqueIndexes().isEmpty()) continue;
            indexMap.put(entity.getTableName(), entity.getUniqueIndexes());
        }
        return indexMap;
    }

    private static void addRenameFieldQueries(Map<String, List<MigrationDataHolder.NameMapping>> renamedFields, List<String> queries) {
        String renameFieldTemplate = "ALTER TABLE %s%nRENAME COLUMN %s TO %s;%n";
        for (Map.Entry<String, List<MigrationDataHolder.NameMapping>> entry : renamedFields.entrySet()) {
            for (MigrationDataHolder.NameMapping nameMapping : entry.getValue()) {
                queries.add(String.format(renameFieldTemplate, entry.getKey(), nameMapping.oldName(), nameMapping.newName()));
            }
        }
    }

    private static void addRenameTableQueries(List<MigrationDataHolder.NameMapping> renamedEntities, List<String> queries) {
        String renameTableTemplate = "RENAME TABLE %s TO %s;%n";
        for (MigrationDataHolder.NameMapping nameMapping : renamedEntities) {
            queries.add(String.format(renameTableTemplate, nameMapping.oldName(), nameMapping.newName()));
        }
    }

    private static void addCreatePrimaryKeyQueries(Set<String> entities, List<String> addedEntities, Module currentModel, List<String> queries) {
        String addPrimaryKeyQuery = "ALTER TABLE %s%nADD PRIMARY KEY (%s);%n";
        for (String tableName : entities) {
            if (addedEntities.contains(tableName)) continue;
            Optional<Entity> entity = currentModel.getEntityByTableName(tableName);
            entity.ifPresent(value -> queries.add(String.format(addPrimaryKeyQuery, tableName, value.getKeys().stream().map(EntityField::getFieldColumnName).reduce((a, b) -> a + ", " + b).orElse(""))));
        }
    }

    private static void addDropPrimaryKeyQueries(Set<String> changedPrimary, List<String> addedEntities, List<String> queries) {
        if (changedPrimary.isEmpty()) {
            return;
        }
        String dropPrimaryKeyTemplate = "ALTER TABLE %s%nDROP PRIMARY KEY;%n";
        changedPrimary.forEach(table -> {
            if (!addedEntities.contains(table)) {
                queries.add(String.format(dropPrimaryKeyTemplate, table));
            }
        });
    }

    private static void addCreateTableQueries(List<String> addedEntities, Module currentModel, List<String> queries) {
        for (String tableName : addedEntities) {
            Optional<Entity> entity = currentModel.getEntityByTableName(tableName);
            if (!entity.isPresent()) continue;
            try {
                queries.add(SqlScriptUtils.generateCreateTableQuery(entity.get(), new HashMap<String, List<String>>(), SqlScriptUtils.getTableNameWithSchema(entity.get(), "mysql"), "mysql") + System.lineSeparator());
            }
            catch (BalException e) {
                errStream.println("ERROR: failed to generate create table query: " + e.getMessage());
            }
        }
    }

    private static void addDropTableQueries(List<String> entities, List<String> queries) {
        for (String entity : entities) {
            String removeTableTemplate = "DROP TABLE %s;%n";
            queries.add(String.format(removeTableTemplate, entity));
        }
    }

    private static void addDropColumnQueries(Map<String, List<String>> map, List<String> queries) {
        for (Map.Entry<String, List<String>> entry : map.entrySet()) {
            String entity = entry.getKey();
            for (String field : entry.getValue()) {
                String removeFieldTemplate = "ALTER TABLE %s%nDROP COLUMN %s;%n";
                queries.add(String.format(removeFieldTemplate, entity, field));
            }
        }
    }

    private static void addDropForeignKeyQueries(Map<String, List<ForeignKey>> map, List<String> queries) {
        for (Map.Entry<String, List<ForeignKey>> entry : map.entrySet()) {
            String entity = entry.getKey();
            for (ForeignKey foreignKey : entry.getValue()) {
                String warningComment = "-- Please verify the foreign key constraint name before executing the query" + System.lineSeparator();
                String removeForeignKeyTemplate = warningComment + "ALTER TABLE %s%nDROP FOREIGN KEY %s;%n";
                queries.add(String.format(removeForeignKeyTemplate, entity, foreignKey.name()));
                HashMap<String, List<String>> fieldMap = new HashMap<String, List<String>>();
                fieldMap.put(entity, foreignKey.columnNames());
                Migrate.addDropColumnQueries(fieldMap, queries);
            }
        }
    }

    private static void addModifyColumnTypeQueries(Map<String, List<EntityField>> map, List<String> queries) {
        for (Map.Entry<String, List<EntityField>> entry : map.entrySet()) {
            String entity = entry.getKey();
            for (EntityField field : entry.getValue()) {
                String fieldType;
                String fieldName = field.getFieldColumnName();
                try {
                    fieldType = SqlScriptUtils.getSqlType(field, "mysql");
                }
                catch (BalException e) {
                    errStream.println("ERROR: data type conversion failed: " + e.getMessage());
                    return;
                }
                String changeTypeTemplate = "ALTER TABLE %s%nMODIFY COLUMN %s %s%s%s;%n";
                queries.add(String.format(changeTypeTemplate, entity, fieldName, fieldType, field.isOptionalType() ? "" : " NOT NULL", field.isDbGenerated() ? " AUTO_INCREMENT" : ""));
            }
        }
    }

    private static void addCreateFieldQueries(Map<String, List<EntityField>> map, List<String> queries) {
        for (Map.Entry<String, List<EntityField>> entry : map.entrySet()) {
            String entity = entry.getKey();
            for (EntityField field : entry.getValue()) {
                String fieldType;
                String fieldName = field.getFieldColumnName();
                try {
                    fieldType = SqlScriptUtils.getSqlType(field, "mysql");
                }
                catch (BalException e) {
                    errStream.println("ERROR: data type conversion failed: " + e.getMessage());
                    return;
                }
                String addFieldTemplate = "ALTER TABLE %s%nADD COLUMN %s %s%s%s;%n";
                queries.add(String.format(addFieldTemplate, entity, fieldName, fieldType, field.isOptionalType() ? "" : " NOT NULL", field.isDbGenerated() ? " AUTO_INCREMENT" : ""));
            }
        }
    }

    private static void addCreateForeignKeyQueries(Map<String, List<ForeignKey>> map, List<String> queries) {
        String addFKTemplate = "ALTER TABLE %s%nADD CONSTRAINT %s FOREIGN KEY (%s) REFERENCES %s(%s);%n";
        for (Map.Entry<String, List<ForeignKey>> entry : map.entrySet()) {
            String entity = entry.getKey();
            for (ForeignKey foreignKey : entry.getValue()) {
                queries.add(String.format(addFKTemplate, entity, foreignKey.name(), foreignKey.columnNames().stream().reduce((a, b) -> a + ", " + b).orElse(""), foreignKey.referenceTable(), foreignKey.referenceColumns().stream().reduce((a, b) -> a + ", " + b).orElse("")));
            }
        }
    }

    private static void addCreateIndexQueries(Map<String, List<Index>> map, List<String> queries) {
        String addIndexTemplate = "CREATE%s INDEX %s ON %s(%s);%n";
        for (Map.Entry<String, List<Index>> entry : map.entrySet()) {
            String entity = entry.getKey();
            for (Index index : entry.getValue()) {
                queries.add(String.format(addIndexTemplate, index.isUnique() ? " UNIQUE" : "", index.getIndexName(), entity, index.getFields().stream().map(EntityField::getFieldColumnName).reduce((a, b) -> a + ", " + b).orElse("")));
            }
        }
    }

    private static void addDropIndexQueries(Map<String, List<Index>> map, List<String> queries) {
        String dropIndexTemplate = "DROP INDEX %s ON %s;%n";
        for (Map.Entry<String, List<Index>> entry : map.entrySet()) {
            String entity = entry.getKey();
            for (Index index : entry.getValue()) {
                queries.add(String.format(dropIndexTemplate, index.getIndexName(), entity));
            }
        }
    }

    public void setParentCmdParser(CommandLine parentCmdParser) {
    }

    public String getName() {
        return "persist";
    }

    public void printLongDesc(StringBuilder out) {
        out.append("Migrate the Ballerina schema into MySQL").append(System.lineSeparator());
        out.append(System.lineSeparator());
    }

    public void printUsage(StringBuilder stringBuilder) {
        stringBuilder.append("  ballerina persist migrate").append(System.lineSeparator());
    }
}

