/*
 * Decompiled with CFR 0.152.
 */
package com.atomikos.datasource.pool;

import com.atomikos.datasource.pool.ConnectionFactory;
import com.atomikos.datasource.pool.ConnectionPoolException;
import com.atomikos.datasource.pool.ConnectionPoolProperties;
import com.atomikos.datasource.pool.CreateConnectionException;
import com.atomikos.datasource.pool.PoolExhaustedException;
import com.atomikos.datasource.pool.XPooledConnection;
import com.atomikos.datasource.pool.XPooledConnectionEventListener;
import com.atomikos.logging.Logger;
import com.atomikos.logging.LoggerFactory;
import com.atomikos.thread.InterruptedExceptionHelper;
import com.atomikos.thread.TaskManager;
import com.atomikos.timing.AlarmTimer;
import com.atomikos.timing.AlarmTimerListener;
import com.atomikos.timing.PooledAlarmTimer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

public abstract class ConnectionPool<ConnectionType>
implements XPooledConnectionEventListener<ConnectionType> {
    private static Logger LOGGER = LoggerFactory.createLogger(ConnectionPool.class);
    private static final int DEFAULT_MAINTENANCE_INTERVAL = 60;
    protected List<XPooledConnection<ConnectionType>> connections = new ArrayList<XPooledConnection<ConnectionType>>();
    private ConnectionFactory<ConnectionType> connectionFactory;
    private ConnectionPoolProperties properties;
    private boolean destroyed;
    private PooledAlarmTimer maintenanceTimer;
    private String name;

    public ConnectionPool(ConnectionFactory<ConnectionType> connectionFactory, ConnectionPoolProperties properties) throws ConnectionPoolException {
        this.connectionFactory = connectionFactory;
        this.properties = properties;
        this.destroyed = false;
        this.name = properties.getUniqueResourceName();
        this.init();
    }

    private void assertNotDestroyed() throws ConnectionPoolException {
        if (this.destroyed) {
            throw new ConnectionPoolException("Pool was already destroyed - you can no longer use it");
        }
    }

    private void init() throws ConnectionPoolException {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.logTrace(this + ": initializing...");
        }
        this.addConnectionsIfMinPoolSizeNotReached();
        this.launchMaintenanceTimer();
    }

    private void launchMaintenanceTimer() {
        int maintenanceInterval = this.properties.getMaintenanceInterval();
        if (maintenanceInterval <= 0) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.logTrace(this + ": using default maintenance interval...");
            }
            maintenanceInterval = 60;
        }
        this.maintenanceTimer = new PooledAlarmTimer(maintenanceInterval * 1000);
        this.maintenanceTimer.addAlarmTimerListener(new AlarmTimerListener(){

            @Override
            public void alarm(AlarmTimer timer) {
                ConnectionPool.this.reapPool();
                ConnectionPool.this.removeConnectionsThatExceededMaxLifetime();
                ConnectionPool.this.addConnectionsIfMinPoolSizeNotReached();
                ConnectionPool.this.removeIdleConnectionsIfMinPoolSizeExceeded();
            }
        });
        TaskManager.SINGLETON.executeTask(this.maintenanceTimer);
    }

    private synchronized void addConnectionsIfMinPoolSizeNotReached() {
        int connectionsToAdd = this.properties.getMinPoolSize() - this.totalSize();
        for (int i = 0; i < connectionsToAdd; ++i) {
            try {
                XPooledConnection<ConnectionType> xpc = this.createPooledConnection();
                this.connections.add(xpc);
                xpc.registerXPooledConnectionEventListener(this);
                continue;
            }
            catch (Exception dbDown) {
                if (!LOGGER.isTraceEnabled()) continue;
                LOGGER.logTrace(this + ": could not establish initial connection", dbDown);
            }
        }
    }

    private XPooledConnection<ConnectionType> createPooledConnection() throws CreateConnectionException {
        XPooledConnection<ConnectionType> xpc = this.connectionFactory.createPooledConnection();
        return xpc;
    }

    protected abstract ConnectionType recycleConnectionIfPossible() throws Exception;

    public ConnectionType borrowConnection() throws CreateConnectionException, PoolExhaustedException, ConnectionPoolException {
        this.assertNotDestroyed();
        ConnectionType ret = null;
        ret = this.findExistingOpenConnectionForCallingThread();
        if (ret == null) {
            ret = this.findOrWaitForAnAvailableConnection();
        }
        return ret;
    }

    private ConnectionType findOrWaitForAnAvailableConnection() throws ConnectionPoolException {
        ConnectionType ret = null;
        long remainingTime = (long)this.properties.getBorrowConnectionTimeout() * 1000L;
        do {
            if ((ret = (ConnectionType)this.retrieveFirstAvailableConnectionAndGrowPoolIfNecessary()) != null) continue;
            remainingTime = this.waitForAtLeastOneAvailableConnection(remainingTime);
            this.assertNotDestroyed();
        } while (ret == null);
        return ret;
    }

    private ConnectionType retrieveFirstAvailableConnectionAndGrowPoolIfNecessary() throws CreateConnectionException {
        ConnectionType ret = this.retrieveFirstAvailableConnection();
        if (ret == null && this.canGrow()) {
            this.growPool();
            ret = this.retrieveFirstAvailableConnection();
        }
        return ret;
    }

    private ConnectionType findExistingOpenConnectionForCallingThread() {
        ConnectionType recycledConnection = null;
        try {
            recycledConnection = this.recycleConnectionIfPossible();
        }
        catch (Exception e2) {
            LOGGER.logDebug(this + ": error while trying to recycle", e2);
        }
        return recycledConnection;
    }

    protected void logCurrentPoolSize() {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.logTrace(this + ": current size: " + this.availableSize() + "/" + this.totalSize());
        }
    }

    private boolean canGrow() {
        return this.totalSize() < this.properties.getMaxPoolSize();
    }

    protected abstract ConnectionType retrieveFirstAvailableConnection();

    private synchronized void growPool() throws CreateConnectionException {
        XPooledConnection<ConnectionType> xpc = this.createPooledConnection();
        this.connections.add(xpc);
        xpc.registerXPooledConnectionEventListener(this);
        this.logCurrentPoolSize();
    }

    private synchronized void removeIdleConnectionsIfMinPoolSizeExceeded() {
        if (this.connections == null || this.properties.getMaxIdleTime() <= 0) {
            return;
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.logTrace(this + ": trying to shrink pool");
        }
        ArrayList<XPooledConnection<ConnectionType>> connectionsToRemove = new ArrayList<XPooledConnection<ConnectionType>>();
        int maxConnectionsToRemove = this.totalSize() - this.properties.getMinPoolSize();
        if (maxConnectionsToRemove > 0) {
            for (int i = 0; i < this.connections.size(); ++i) {
                XPooledConnection<ConnectionType> xpc = this.connections.get(i);
                long lastRelease = xpc.getLastTimeReleased();
                long maxIdle = this.properties.getMaxIdleTime();
                long now = System.currentTimeMillis();
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.logTrace(this + ": connection idle for " + (now - lastRelease) + "ms");
                }
                if (!xpc.isAvailable() || now - lastRelease < maxIdle * 1000L || connectionsToRemove.size() >= maxConnectionsToRemove) continue;
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.logTrace(this + ": connection idle for more than " + maxIdle + "s, closing it: " + xpc);
                }
                this.destroyPooledConnection(xpc, false);
                connectionsToRemove.add(xpc);
            }
        }
        this.connections.removeAll(connectionsToRemove);
        this.logCurrentPoolSize();
    }

    protected void destroyPooledConnection(XPooledConnection<ConnectionType> xpc, boolean reap) {
        xpc.destroy(reap);
    }

    public synchronized void reapPool() {
        long maxInUseTime = this.properties.getReapTimeout();
        if (this.connections == null || maxInUseTime <= 0L) {
            return;
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.logTrace(this + ": reaping old connections");
        }
        Iterator<XPooledConnection<ConnectionType>> it = this.connections.iterator();
        while (it.hasNext()) {
            XPooledConnection<ConnectionType> xpc = it.next();
            long lastTimeReleased = xpc.getLastTimeAcquired();
            boolean inUse = !xpc.isAvailable();
            long now = System.currentTimeMillis();
            if (!inUse || now - maxInUseTime * 1000L <= lastTimeReleased) continue;
            if (LOGGER.isTraceEnabled()) {
                LOGGER.logTrace(this + ": connection in use for more than " + maxInUseTime + "s, reaping it: " + xpc);
            }
            xpc.destroy(true);
            it.remove();
        }
        this.logCurrentPoolSize();
    }

    private synchronized void removeConnectionsThatExceededMaxLifetime() {
        long maxLifetime = (long)this.properties.getMaxLifetime() * 1000L;
        if (this.connections == null || maxLifetime <= 0L) {
            return;
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.logTrace(this + ": closing connections that exceeded maxLifetime");
        }
        Iterator<XPooledConnection<ConnectionType>> it = this.connections.iterator();
        long now = System.currentTimeMillis();
        while (it.hasNext()) {
            XPooledConnection<ConnectionType> xpc = it.next();
            long creationTime = xpc.getCreationTime();
            if (!xpc.isAvailable() || now - creationTime < maxLifetime) continue;
            if (LOGGER.isTraceEnabled()) {
                LOGGER.logTrace(this + ": connection in use for more than maxLifetime, destroying it: " + xpc);
            }
            this.destroyPooledConnection(xpc, false);
            it.remove();
        }
        this.logCurrentPoolSize();
    }

    public synchronized void destroy() {
        if (!this.destroyed) {
            LOGGER.logInfo(this + ": destroying pool...");
            for (int i = 0; i < this.connections.size(); ++i) {
                XPooledConnection<ConnectionType> xpc = this.connections.get(i);
                if (!xpc.isAvailable()) {
                    LOGGER.logWarning(this + ": connection is still in use on pool destroy: " + xpc + " - please check your shutdown sequence to avoid heuristic termination of ongoing transactions!");
                }
                this.destroyPooledConnection(xpc, false);
            }
            this.connections = null;
            this.destroyed = true;
            this.maintenanceTimer.stopTimer();
            if (LOGGER.isTraceEnabled()) {
                LOGGER.logTrace(this + ": pool destroyed.");
            }
        }
    }

    public synchronized void refresh() {
        ArrayList<XPooledConnection<ConnectionType>> connectionsToRemove = new ArrayList<XPooledConnection<ConnectionType>>();
        for (XPooledConnection<ConnectionType> conn : this.connections) {
            if (!conn.isAvailable()) continue;
            connectionsToRemove.add(conn);
            this.destroyPooledConnection(conn, false);
        }
        this.connections.removeAll(connectionsToRemove);
        this.addConnectionsIfMinPoolSizeNotReached();
    }

    private synchronized long waitForAtLeastOneAvailableConnection(long waitTime) throws PoolExhaustedException {
        while (this.availableSize() == 0) {
            long before;
            block6: {
                if (waitTime <= 0L) {
                    throw new PoolExhaustedException("ConnectionPool: pool is empty - increase either maxPoolSize or borrowConnectionTimeout");
                }
                before = System.currentTimeMillis();
                try {
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.logTrace(this + ": about to wait for connection during " + waitTime + "ms...");
                    }
                    this.wait(waitTime);
                }
                catch (InterruptedException ex) {
                    InterruptedExceptionHelper.handleInterruptedException(ex);
                    if (!LOGGER.isTraceEnabled()) break block6;
                    LOGGER.logTrace(this + ": interrupted during wait", ex);
                }
            }
            if (LOGGER.isTraceEnabled()) {
                LOGGER.logTrace(this + ": done waiting.");
            }
            long now = System.currentTimeMillis();
            waitTime -= now - before;
        }
        return waitTime;
    }

    public synchronized int availableSize() {
        int ret = 0;
        if (!this.destroyed) {
            int count2 = 0;
            for (int i = 0; i < this.connections.size(); ++i) {
                XPooledConnection<ConnectionType> xpc = this.connections.get(i);
                if (!xpc.isAvailable()) continue;
                ++count2;
            }
            ret = count2;
        }
        return ret;
    }

    public synchronized int totalSize() {
        if (this.destroyed) {
            return 0;
        }
        return this.connections.size();
    }

    @Override
    public synchronized void onXPooledConnectionTerminated(XPooledConnection<ConnectionType> connection) {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.logTrace(this + ": connection " + connection + " became available, notifying potentially waiting threads");
        }
        this.notify();
    }

    public String toString() {
        return "atomikos connection pool '" + this.name + "'";
    }
}

