/*
 * Decompiled with CFR 0.152.
 */
package io.ballerina.graphql.cmd;

import io.ballerina.cli.BLauncherCmd;
import io.ballerina.graphql.cmd.Utils;
import io.ballerina.graphql.cmd.pojo.Config;
import io.ballerina.graphql.cmd.pojo.Project;
import io.ballerina.graphql.exception.CmdException;
import io.ballerina.graphql.exception.ParseException;
import io.ballerina.graphql.exception.ValidationException;
import io.ballerina.graphql.generator.GraphqlProject;
import io.ballerina.graphql.generator.client.GraphqlClientProject;
import io.ballerina.graphql.generator.client.exception.ClientCodeGenerationException;
import io.ballerina.graphql.generator.client.generator.ClientCodeGenerator;
import io.ballerina.graphql.generator.client.pojo.Extension;
import io.ballerina.graphql.generator.service.GraphqlServiceProject;
import io.ballerina.graphql.generator.service.diagnostic.ServiceDiagnosticMessages;
import io.ballerina.graphql.generator.service.exception.ServiceGenerationException;
import io.ballerina.graphql.generator.service.generator.ServiceCodeGenerator;
import io.ballerina.graphql.schema.diagnostic.DiagnosticMessages;
import io.ballerina.graphql.schema.exception.SchemaFileGenerationException;
import io.ballerina.graphql.schema.generator.SdlSchemaGenerator;
import io.ballerina.graphql.validator.ConfigValidator;
import io.ballerina.graphql.validator.QueryValidator;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.BaseConstructor;
import org.yaml.snakeyaml.constructor.Constructor;
import org.yaml.snakeyaml.error.YAMLException;
import picocli.CommandLine;

@CommandLine.Command(name="graphql", description={"Generates Ballerina clients for GraphQL queries with GraphQL SDL, Ballerina services for GraphQL schema and SDL schema for the given Ballerina GraphQL service."})
public class GraphqlCmd
implements BLauncherCmd {
    private static final int EXIT_CODE_0 = 0;
    private static final int EXIT_CODE_1 = 1;
    private static final int EXIT_CODE_2 = 2;
    private static final String CMD_NAME = "graphql";
    private static final ExitHandler DEFAULT_EXIT_HANDLER = code -> Runtime.getRuntime().exit(code);
    private final PrintStream outStream;
    private final Path executionPath;
    private final ExitHandler exitHandler;
    @CommandLine.Option(names={"-h", "--help"}, hidden=true)
    private boolean helpFlag;
    @CommandLine.Option(names={"-i", "--input"}, description={"File path to the GraphQL configuration file, GraphQL schema file or Ballerina service file."})
    private String inputPath;
    @CommandLine.Option(names={"-o", "--output"}, description={"Directory to store the generated Ballerina clients, Ballerina services or SDL schema file. If this is not provided, the generated files will be stored in the current execution directory."})
    private String outputPath;
    @CommandLine.Option(names={"-s", "--service"}, description={"Base path of the service that the SDL schema is needed to be generated. If this is not provided, generate the SDL schema for each GraphQL service in the source file."})
    private String serviceBasePath;
    @CommandLine.Option(names={"-m", "--mode"}, description={"Ballerina operation mode. It can be client, service or schema."})
    private String mode;
    @CommandLine.Option(names={"-r", "--use-records-for-objects"}, description={"Inform the generator to generate records types where ever possible"})
    private boolean useRecordsForObjectsFlag;
    private ClientCodeGenerator clientCodeGenerator;
    private ServiceCodeGenerator serviceCodeGenerator;

    public GraphqlCmd() {
        this(System.err, Paths.get(System.getProperty("user.dir"), new String[0]));
    }

    public GraphqlCmd(PrintStream outStream, Path executionDir) {
        this(outStream, executionDir, DEFAULT_EXIT_HANDLER);
    }

    public GraphqlCmd(PrintStream outStream, Path executionDir, ExitHandler exitHandler) {
        this.outStream = outStream;
        this.executionPath = executionDir;
        this.exitHandler = exitHandler;
    }

    private void exit(int code) {
        this.exitHandler.exit(code);
    }

    public void execute() {
        try {
            if (this.helpFlag) {
                this.printLongDesc(new StringBuilder());
                this.outStream.flush();
                this.exit(0);
                return;
            }
            if (this.inputPath == null || this.inputPath.isEmpty()) {
                this.printLongDesc(new StringBuilder());
                this.outStream.flush();
                this.exit(2);
                return;
            }
            this.validateInputFlags();
            this.executeOperation();
        }
        catch (CmdException | ParseException | ValidationException | ClientCodeGenerationException | ServiceGenerationException | SchemaFileGenerationException | IOException e) {
            this.outStream.println(e.getMessage());
            this.exit(1);
            return;
        }
        this.exit(0);
    }

    private void validateInputFlags() throws CmdException {
        if (!this.validInputFileExtension(this.inputPath)) {
            throw new CmdException(String.format("File \"%s\" is invalid. Supported input files are,\nA GraphQL configuration file with .yaml/.yml extension, \na Ballerina service file with .bal extension or \na GraphQL schema file with .graphql extension. \nPlease provide the path of the input  file with -i or --input flag.\ne.g: bal graphql --input <GraphQL configuration file>", this.inputPath));
        }
        if (!this.isModeCompatible()) {
            throw new CmdException(String.format("\"%s\" mode is not allowed to used with file \"%s\".\nThe mode flag value should be client, service or schema. Input file should be a GraphQL configuration file with .yaml/.yml extension, a Ballerina service file with .bal or a GraphQL schema file with .graphql respectively.", this.mode, this.inputPath));
        }
        if (this.useRecordsForObjectsFlag && !this.inputPath.endsWith(".graphql")) {
            throw new CmdException(String.format("The use-records-for-objects flag is incompatible with: \"%s\"", this.mode));
        }
    }

    private boolean validInputFileExtension(String filePath) {
        return filePath.endsWith(".yaml") || filePath.endsWith(".yml") || filePath.endsWith(".bal") || filePath.endsWith(".graphql");
    }

    private boolean isModeCompatible() throws CmdException {
        if (this.mode != null) {
            if ("client".equals(this.mode)) {
                return this.inputPath.endsWith(".yaml") || this.inputPath.endsWith(".yml");
            }
            if ("schema".equals(this.mode)) {
                return this.inputPath.endsWith(".bal");
            }
            if ("service".equals(this.mode)) {
                return this.inputPath.endsWith(".graphql");
            }
            throw new CmdException(String.format("\"%s\" is not a supported argument for mode flag. The mode flag argument should be one of these \"client\", \"service\" or \"schema\"", this.mode));
        }
        return true;
    }

    private void executeOperation() throws CmdException, ParseException, IOException, ValidationException, ClientCodeGenerationException, SchemaFileGenerationException, ServiceGenerationException {
        if (("client".equals(this.mode) || this.mode == null) && (this.inputPath.endsWith(".yaml") || this.inputPath.endsWith(".yml"))) {
            this.setClientCodeGenerator(new ClientCodeGenerator());
            this.generateClient(this.inputPath);
        } else if (("schema".equals(this.mode) || this.mode == null) && this.inputPath.endsWith(".bal")) {
            this.generateSchema(this.inputPath);
        } else if (("service".equals(this.mode) || this.mode == null) && this.inputPath.endsWith(".graphql")) {
            this.setServiceCodeGenerator(new ServiceCodeGenerator());
            this.generateService(this.inputPath);
        }
    }

    private void generateClient(String filePath) throws ParseException, IOException, ValidationException, ClientCodeGenerationException {
        Config config = this.readConfig(filePath);
        ConfigValidator.getInstance().validate(config);
        List<GraphqlClientProject> projects = this.populateProjects(config);
        for (GraphqlClientProject project : projects) {
            Utils.validateGraphqlProject((GraphqlProject)project);
            QueryValidator.getInstance().validate(project);
        }
        for (GraphqlClientProject project : projects) {
            this.clientCodeGenerator.generate((GraphqlProject)project);
        }
    }

    private void generateService(String filePath) throws IOException, ValidationException, ServiceGenerationException {
        File graphqlFile = new File(filePath);
        if (!graphqlFile.exists()) {
            throw new ServiceGenerationException(ServiceDiagnosticMessages.GRAPHQL_SERVICE_GEN_100, null, new String[]{String.format("Provided Schema file \"%s\" does not exist.", filePath)});
        }
        if (!graphqlFile.canRead()) {
            throw new ServiceGenerationException(ServiceDiagnosticMessages.GRAPHQL_SERVICE_GEN_100, null, new String[]{String.format("Provided Schema file \"%s\" is not allowed to be read", filePath)});
        }
        GraphqlServiceProject graphqlProject = new GraphqlServiceProject("root", filePath, this.getTargetOutputPath().toString());
        Utils.validateGraphqlProject((GraphqlProject)graphqlProject);
        if (this.useRecordsForObjectsFlag) {
            this.serviceCodeGenerator.enableToUseRecords();
        }
        this.serviceCodeGenerator.generate((GraphqlProject)graphqlProject);
    }

    private void generateSchema(String fileName) throws SchemaFileGenerationException {
        File balFile = new File(fileName);
        if (!balFile.exists()) {
            throw new SchemaFileGenerationException(DiagnosticMessages.SDL_SCHEMA_103, null, new String[]{"Provided Ballerina file path does not exist"});
        }
        if (!balFile.canRead()) {
            throw new SchemaFileGenerationException(DiagnosticMessages.SDL_SCHEMA_103, null, new String[]{"Cannot read provided Ballerina file (Permission denied)"});
        }
        Path balFilePath = null;
        try {
            balFilePath = Paths.get(balFile.getCanonicalPath(), new String[0]);
        }
        catch (IOException e) {
            throw new SchemaFileGenerationException(DiagnosticMessages.SDL_SCHEMA_103, null, new String[]{e.toString()});
        }
        SdlSchemaGenerator.generate((Path)balFilePath, (Path)this.getTargetOutputPath(), (String)this.serviceBasePath, (PrintStream)this.outStream);
    }

    private Config readConfig(String filePath) throws FileNotFoundException, ParseException {
        try {
            FileInputStream inputStream = new FileInputStream(new File(filePath));
            Constructor constructor = Utils.getProcessedConstructor();
            Yaml yaml = new Yaml((BaseConstructor)constructor);
            Config config = (Config)yaml.load((InputStream)inputStream);
            if (config == null) {
                throw new ParseException("The GraphQL configuration YAML file is empty. \nPlease provide a valid content in the YAML file.");
            }
            return config;
        }
        catch (YAMLException e) {
            throw new ParseException("The GraphQL config file is not in the specific config file format.\n" + e.getMessage());
        }
    }

    private List<GraphqlClientProject> populateProjects(Config config) {
        ArrayList<GraphqlClientProject> graphqlClientProjects = new ArrayList<GraphqlClientProject>();
        String schema = config.getSchema();
        List<String> documents = config.getDocuments();
        Extension extensions = config.getExtensions();
        Map<String, Project> projects = config.getProjects();
        if (schema != null || documents != null || extensions != null) {
            graphqlClientProjects.add(new GraphqlClientProject("root", schema, documents, extensions, this.getTargetOutputPath().toString()));
        }
        if (projects != null) {
            for (String projectName : projects.keySet()) {
                graphqlClientProjects.add(new GraphqlClientProject(projectName, projects.get(projectName).getSchema(), projects.get(projectName).getDocuments(), projects.get(projectName).getExtensions(), this.getTargetOutputPath().toString()));
            }
        }
        return graphqlClientProjects;
    }

    private Path getTargetOutputPath() {
        Path targetOutputPath = this.executionPath;
        if (this.outputPath != null) {
            targetOutputPath = Paths.get(this.outputPath, new String[0]).isAbsolute() ? Paths.get(this.outputPath, new String[0]) : Paths.get(targetOutputPath.toString(), this.outputPath);
        }
        return targetOutputPath;
    }

    public void setClientCodeGenerator(ClientCodeGenerator clientCodeGenerator) {
        this.clientCodeGenerator = clientCodeGenerator;
    }

    public void setServiceCodeGenerator(ServiceCodeGenerator serviceCodeGenerator) {
        this.serviceCodeGenerator = serviceCodeGenerator;
    }

    public String getName() {
        return CMD_NAME;
    }

    public void printLongDesc(StringBuilder stringBuilder) {
        Class<GraphqlCmd> cmdClass = GraphqlCmd.class;
        ClassLoader classLoader = cmdClass.getClassLoader();
        InputStream inputStream = classLoader.getResourceAsStream("ballerina-graphql.help");
        try (InputStreamReader inputStreamREader = new InputStreamReader(inputStream, StandardCharsets.UTF_8);
             BufferedReader br = new BufferedReader(inputStreamREader);){
            String content = br.readLine();
            this.outStream.append(content);
            while ((content = br.readLine()) != null) {
                this.outStream.append('\n').append(content);
            }
            this.outStream.append('\n');
        }
        catch (IOException ex) {
            throw new IllegalStateException(ex);
        }
    }

    public void printUsage(StringBuilder stringBuilder) {
    }

    public void setParentCmdParser(CommandLine commandLine) {
    }

    @FunctionalInterface
    public static interface ExitHandler {
        public void exit(int var1);
    }
}

