// Copyright (c) 2025 WSO2 LLC (http://www.wso2.com).
//
// WSO2 LLC. licenses this file to you under the Apache License,
// Version 2.0 (the "License"); you may not use this file except
// in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

import ballerina/http;

# Represents a request message for the chat service.
#
# + sessionId - A unique identifier for the chat session.
# + message - The content of the chat message sent by the user.
public type ChatReqMessage record {|
    string sessionId;
    string message;
|};

# Represents a response message from the chat service.
#
# + message - The response message generated by the chat service.
public type ChatRespMessage record {|
    string message;
|};

# Represents the configuration for a chat client.
public type ChatClientConfiguration record {|
    *http:ClientConfiguration;
|};

# Defines a chat service interface that handles incoming chat messages.
public type ChatService distinct service object {
    *http:Service;
    resource function post chat(@http:Payload ChatReqMessage request) returns ChatRespMessage|error;
};

type ExecutionTrace record {|
    (ExecutionResult|ExecutionError|Error)[] steps;
    string answer?;
|};

# Configurations for controlling the behaviours when communicating with a remote HTTP endpoint.
@display {label: "Connection Configuration"}
public type ConnectionConfig record {|

    # The HTTP version understood by the client
    @display {label: "HTTP Version"}
    http:HttpVersion httpVersion = http:HTTP_2_0;

    # Configurations related to HTTP/1.x protocol
    @display {label: "HTTP1 Settings"}
    http:ClientHttp1Settings http1Settings = {};

    # Configurations related to HTTP/2 protocol
    @display {label: "HTTP2 Settings"}
    http:ClientHttp2Settings http2Settings = {};

    # The maximum time to wait (in seconds) for a response before closing the connection
    @display {label: "Timeout"}
    decimal timeout = 60;

    # The choice of setting `forwarded`/`x-forwarded` header
    @display {label: "Forwarded"}
    string forwarded = "disable";

    # Configurations associated with request pooling
    @display {label: "Pool Configuration"}
    http:PoolConfiguration poolConfig?;

    # HTTP caching related configurations
    @display {label: "Cache Configuration"}
    http:CacheConfig cache = {};

    # Specifies the way of handling compression (`accept-encoding`) header
    @display {label: "Compression"}
    http:Compression compression = http:COMPRESSION_AUTO;

    # Configurations associated with the behaviour of the Circuit Breaker
    @display {label: "Circuit Breaker Configuration"}
    http:CircuitBreakerConfig circuitBreaker?;

    # Configurations associated with retrying
    @display {label: "Retry Configuration"}
    http:RetryConfig retryConfig?;

    # Configurations associated with inbound response size limits
    @display {label: "Response Limit Configuration"}
    http:ResponseLimitConfigs responseLimits = {};

    # SSL/TLS-related options
    @display {label: "Secure Socket Configuration"}
    http:ClientSecureSocket secureSocket?;

    # Proxy server related options
    @display {label: "Proxy Configuration"}
    http:ProxyConfig proxy?;

    # Enables the inbound payload validation functionality which provided by the constraint package. Enabled by default
    @display {label: "Payload Validation"}
    boolean validation = true;
|};

# Represent the automatic chunker configuration in the `VectorKnowledgeBase`.
# This chunker will be chosen automatically based on the document type.
public const AUTO = "AUTO";

# Represents the disabled chunker configuration in the `VectorKnowledgeBase`.
# This chunker will not perform any chunking and will treat the entire document as a single chunk.
public const DISABLE = "DISABLE";

# Represents the configuration for the `ModelProvider.generate()` method.
public type GeneratorConfig record {|
    # Configuration for retrying on response parsing failures
    @display {label: "Retry Configuration"}
    RetryConfig retryConfig?;
|};

# Represents retry configuration on response parsing failures.
public type RetryConfig record {|
    # Number of retry attempts
    @display {label: "Retry Count"}
    int count = 0;

    # Retry interval in seconds
    @display {label: "Retry Interval"}
    decimal interval?;
|};
