// Copyright (c) 2025 WSO2 LLC (http://www.wso2.com).
//
// WSO2 LLC. licenses this file to you under the Apache License,
// Version 2.0 (the "License"); you may not use this file except
// in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

import ballerina/math.vector;
import ballerina/uuid;

# Represents a vector store that provides persistence, management, and search capabilities for vector embeddings.
public type VectorStore distinct isolated object {

    # Adds vector entries to the store.
    #
    # + entries - The array of vector entries to add
    # + return - An `ai:Error` if the operation fails; otherwise, `nil`
    public isolated function add(VectorEntry[] entries) returns Error?;

    # Searches for vectors in the store that are most similar to a given query.
    #
    # + query - The vector store query that specifies the search criteria
    # + return - An array of matching vectors with their similarity scores,
    # or an `ai:Error` if the operation fails
    public isolated function query(VectorStoreQuery query) returns VectorMatch[]|Error;

    # Deletes a vector entry from the store by its unique ID.
    #
    # + ids - The unique identifiers of the vector entries to delete
    # + return - An `ai:Error` if the operation fails; otherwise, `nil`
    public isolated function delete(string|string[] ids) returns Error?;
};

# An in-memory vector store implementation that provides simple storage for vector entries.
public distinct isolated class InMemoryVectorStore {
    *VectorStore;
    private final table<InMemoryVectorEntry> key(id) entries = table [];
    private final SimilarityMetric similarityMetric;

    # Initializes a new in-memory vector store.
    #
    # + similarityMetric - The metric used for vector similarity
    public isolated function init(SimilarityMetric similarityMetric = COSINE) returns Error? {
        self.similarityMetric = similarityMetric;
    }

    # Adds vector entries to the in-memory store.
    # Only supports dense vectors in this implementation. 
    # If a vector entry with the same ID already exists, it will be replaced.
    #
    # + entries - Array of vector entries to store. If an entry does not contain an id,
    # a unique id will be generated by default.
    # + return - `nil` on success; an Error if non-dense vectors are provided
    public isolated function add(VectorEntry[] entries) returns Error? {
        foreach VectorEntry entry in entries {
            if entry.embedding !is Vector {
                return error Error("InMemoryVectorStore supports dense vectors exclusively");
            }
            if entry.id is () {
                entry.id = uuid:createRandomUuid();
            }
        }
        readonly & VectorEntry[] clonedEntries = entries.cloneReadOnly();
        lock {
            foreach VectorEntry entry in clonedEntries {
                self.entries.put({id: <string>entry.id, embedding: entry.embedding, chunk: entry.chunk});
            }
        }
    }

    # Queries the vector store for vectors similar to the given query.
    #
    # + query - The query containing the embedding vector and optional filters
    # + return - An array of vector matches sorted by similarity score (limited to topK), 
    # or an `ai:Error` if the query fails
    public isolated function query(VectorStoreQuery query) returns VectorMatch[]|Error {
        lock {
            VectorStoreQuery clonedQuery = query.cloneReadOnly();
            Embedding? embedding = clonedQuery.embedding;
            MetadataFilters? filters = clonedQuery.filters;
            if embedding !is Vector? {
                return error Error("InMemoryVectorStore supports dense vectors exclusively");
            }
            if embedding is () && filters is () {
                stream<VectorMatch> matchStream = from InMemoryVectorEntry entry in self.entries
                    select {
                        chunk: entry.chunk.cloneReadOnly(),
                        embedding: entry.embedding.cloneReadOnly(),
                        similarityScore: 0.0,
                        id: entry.id
                    };
                return self.getEntries(matchStream, query.topK).cloneReadOnly();
            }
            if embedding is () && filters !is () {
                MetadataFilters metadataFilters = filters.cloneReadOnly();
                stream<VectorMatch, Error?> matchStream = from InMemoryVectorEntry entry in self.entries
                    where check entryMatchesFilters(entry, metadataFilters)
                    select {
                        chunk: entry.chunk.cloneReadOnly(),
                        embedding: entry.embedding.cloneReadOnly(),
                        similarityScore: 0.0,
                        id: entry.id
                    };
                return self.getEntries(matchStream, query.topK).cloneReadOnly();
            }
            if filters !is () {
                final readonly & MetadataFilters metadataFilters = filters.cloneReadOnly();
                stream<VectorMatch, Error?> matchStream = from InMemoryVectorEntry entry in self.entries
                    let float similarity = self.calculateSimilarity(<Vector>query.embedding.clone(), <Vector>entry.embedding)
                    where check entryMatchesFilters(entry, metadataFilters)
                    order by similarity descending
                    select {
                        chunk: entry.chunk.cloneReadOnly(),
                        embedding: entry.embedding.cloneReadOnly(),
                        similarityScore: similarity,
                        id: entry.id
                    };
                return self.getEntries(matchStream, query.topK).cloneReadOnly();
            }
            stream<VectorMatch, Error?> matchStream = from InMemoryVectorEntry entry in self.entries
                let float similarity = self.calculateSimilarity(<Vector>query.embedding.clone(), <Vector>entry.embedding)
                order by similarity descending
                select {
                    chunk: entry.chunk,
                    embedding: entry.embedding,
                    similarityScore: similarity,
                    id: entry.id
                };
            return self.getEntries(matchStream, query.topK).cloneReadOnly();
        } on fail error err {
            return error("Failed to query vector store", err);
        }
    }

    private isolated function getEntries(stream<VectorMatch, Error?> entryStream, int topK) returns VectorMatch[]|Error {
        if topK < 1 {
            // if topK is a negative or zero then return all entries
            return from VectorMatch entry in entryStream
                select entry;
        }
        return from VectorMatch entry in entryStream
            limit topK
            select entry;
    }

    private isolated function calculateSimilarity(Vector queryEmbedding, Vector entryEmbedding) returns float {
        match self.similarityMetric {
            COSINE => {
                return vector:cosineSimilarity(queryEmbedding, entryEmbedding);
            }
            EUCLIDEAN => {
                return vector:euclideanDistance(queryEmbedding, entryEmbedding);
            }
            DOT_PRODUCT => {
                return vector:dotProduct(queryEmbedding, entryEmbedding);
            }
        }
        return vector:cosineSimilarity(queryEmbedding, entryEmbedding);
    }

    # Deletes a vector entry from the in-memory store.
    # Remove entries that matches the given reference identifiers.
    #
    # + ids - The reference identifiers of vector entries to delete
    # + return - `ai:Error` if the reference ID is not found, otherwise `nil`
    public isolated function delete(string|string[] ids) returns Error? {
        if ids is string {
            check self.deleteEntry(ids);
            return;
        }
        _ = ids.'map(id => check self.deleteEntry(id));
    }

    private isolated function deleteEntry(string id) returns Error? {
        lock {
            if !self.entries.hasKey(id) {
                return error Error(string `Vector entry with reference id '${id}' not found`);
            }
            _ = self.entries.remove(id);
        }
    }
}
